<?php
/**
 * Plugin Name: Ahur System
 * Plugin URI: https://khaleghi.info
 * Description: Core system plugin for WordPress repository integration and updates management.
 * Version: 0.0.10
 * Author: Ahur Team
 * Author URI: https://khaleghi.info
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: ahur-system
 * Domain Path: /languages
 * Network: false
 * Requires at least: 5.0
 * Requires PHP: 7.4
 * Tested up to: 6.4
 *
 * @package AhurSystem
 * @author Ahur Team
 * @version 0.0.10
 */

// Prevent direct access
if (!defined("ABSPATH")) {
    exit();
}

// Define plugin constants
define("AHUR_SYSTEM_VERSION", "0.0.10");
define("AHUR_SYSTEM_PLUGIN_DIR", plugin_dir_path(__FILE__));
define("AHUR_SYSTEM_PLUGIN_URL", plugin_dir_url(__FILE__));
define("AHUR_SYSTEM_API_BASE", "https://wordpress.khaleghi.info/api");

/**
 * Main Ahur System Plugin Class
 */
class AhurSystem
{
    /**
     * Singleton instance
     */
    private static $instance = null;

    /**
     * Get singleton instance
     */
    public static function getInstance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct()
    {
        add_action("plugins_loaded", [$this, "init"]);
        register_activation_hook(__FILE__, [$this, "activate"]);
        register_deactivation_hook(__FILE__, [$this, "deactivate"]);

        // Prevent deletion
        add_filter(
            "plugin_action_links_" . plugin_basename(__FILE__),
            [$this, "preventDeletion"],
            10,
            4,
        );
    }

    /**
     * Initialize plugin
     */
    public function init()
    {
        // Load text domain
        load_plugin_textdomain(
            "ahur-system",
            false,
            dirname(plugin_basename(__FILE__)) . "/languages",
        );

        // Initialize system update controller
        $this->initSystemUpdate();

        // Filter plugin name for translation
        add_filter("plugin_row_meta", [$this, "translatePluginName"], 10, 4);

        // Add JavaScript for plugin activation retry
        add_action("admin_footer", [$this, "addActivationRetryScript"]);

        // Add JavaScript for plugin install page button states
        add_action("admin_footer", [$this, "addPluginInstallPageScript"]);

        // Add custom CSS for button states
        add_action("admin_head", [$this, "addPluginInstallStyles"]);

        // Add settings page under Settings menu
        add_action('admin_menu', [$this, 'addSettingsMenu']);

        // Register settings
        add_action('admin_init', [$this, 'registerSettings']);

        // Add debug action for troubleshooting plugin slugs
        add_action('admin_init', [$this, 'debugPluginSlugs']);

        // Admin AJAX handlers
        if (is_admin()) {
            add_action("wp_ajax_ahur_clear_cache", [$this, "ajaxClearCache"]);
            add_action("wp_ajax_ahur_force_check", [$this, "ajaxForceCheck"]);
            add_action("wp_ajax_ahur_test_connection", [
                $this,
                "ajaxTestConnection",
            ]);
            add_action("wp_ajax_ahur_check_plugin_status", [$this, "ajaxCheckPluginStatus"]);
        }
    }

    /**
     * Initialize system update controller
     */
    private function initSystemUpdate()
    {
        require_once AHUR_SYSTEM_PLUGIN_DIR . "includes/system-update.php";

        // Don't create object yet - let it initialize on 'init' action
        new AhurSystemUpdate();
    }

    /**
     * Prevent plugin deletion
     */
    public function preventDeletion(
        $actions,
        $plugin_file,
        $plugin_data,
        $context,
    ) {
        // Remove deactivate and delete actions
        unset($actions["deactivate"]);
        unset($actions["delete"]);

        // Add custom message
        $actions["protected"] =
            '<span style="color: #666;">' .
            __("Protected System Plugin", "ahur-system") .
            "</span>";

        return $actions;
    }

    /**
     * Translate plugin name in plugins list
     */
    public function translatePluginName(
        $plugin_meta,
        $plugin_file,
        $plugin_data,
        $status,
    ) {
        if ($plugin_file === plugin_basename(__FILE__)) {
            // Add translated name as meta (this helps with display)
            $plugin_meta["translated_name"] = __("Ahur System", "ahur-system");
        }
        return $plugin_meta;
    }

    /**
     * Plugin activation
     */
    public function activate()
    {
        // Add activation notice
        add_option("ahur_system_activated", true);

        // Schedule daily check for updates
        if (!wp_next_scheduled("ahur_system_check_updates")) {
            wp_schedule_event(time(), "daily", "ahur_system_check_updates");
        }
    }

    /**
     * Plugin deactivation
     */
    public function deactivate()
    {
        // Clear scheduled events
        wp_clear_scheduled_hook("ahur_system_check_updates");

        // Clear caches
        $this->clearAllCaches();
    }



    /**
     * Add settings menu under Settings
     */
    public function addSettingsMenu()
    {
        add_options_page(
            __('Ahur System', 'ahur-system'),
            __('سامانه اهور', 'ahur-system'),
            'manage_options',
            'ahur-system',
            [$this, 'adminPage']
        );
    }

    /**
     * Register settings
     */
    public function registerSettings()
    {
        register_setting("ahur_system_settings", "ahur_system_api_url");
        register_setting("ahur_system_settings", "ahur_system_enable_updates");
        register_setting("ahur_system_settings", "ahur_system_cache_time");
    }

    /**
     * Admin page
     */
    public function adminPage()
    {
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>

            <form method="post" action="options.php">
                <?php
                settings_fields("ahur_system_settings");
                do_settings_sections("ahur_system_settings");
                ?>

            <div class="card">
                <h2><?php _e("Repository Integration", "ahur-system"); ?></h2>
                <p><?php _e(
                    "Ahur System integrates your WordPress site with Khaleghi repository for seamless plugin and theme updates.",
                    "ahur-system",
                ); ?></p>

                <table class="form-table" role="presentation">
                    <tbody>
                        <tr>
                            <th scope="row">
                                <label for="ahur_system_api_url"><?php _e(
                                    "API Base URL",
                                    "ahur-system",
                                ); ?></label>
                            </th>
                            <td>
                                <input type="url" id="ahur_system_api_url" name="ahur_system_api_url"
                                       value="<?php echo esc_attr(
                                           get_option(
                                               "ahur_system_api_url",
                                               AHUR_SYSTEM_API_BASE,
                                           ),
                                       ); ?>"
                                       class="regular-text" />
                                <p class="description"><?php _e(
                                    "Base URL for the repository API.",
                                    "ahur-system",
                                ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <label for="ahur_system_enable_updates"><?php _e(
                                    "Enable Updates",
                                    "ahur-system",
                                ); ?></label>
                            </th>
                            <td>
                                <input type="checkbox" id="ahur_system_enable_updates" name="ahur_system_enable_updates"
                                       value="1" <?php checked(
                                           get_option(
                                               "ahur_system_enable_updates",
                                               1,
                                           ),
                                       ); ?> />
                                <label for="ahur_system_enable_updates"><?php _e(
                                    "Enable automatic update checks",
                                    "ahur-system",
                                ); ?></label>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <label for="ahur_system_cache_time"><?php _e(
                                    "Cache Time",
                                    "ahur-system",
                                ); ?></label>
                            </th>
                            <td>
                                <input type="number" id="ahur_system_cache_time" name="ahur_system_cache_time"
                                       value="<?php echo esc_attr(
                                           get_option(
                                               "ahur_system_cache_time",
                                               6,
                                           ),
                                       ); ?>"
                                       min="1" max="24" />
                                <span class="description"><?php _e(
                                    "hours",
                                    "ahur-system",
                                ); ?></span>
                                <p class="description"><?php _e(
                                    "How long to cache API responses.",
                                    "ahur-system",
                                ); ?></p>
                            </td>
                        </tr>
                    </tbody>
                </table>
            </div>

            <div class="card">
                <h2><?php _e("System Information", "ahur-system"); ?></h2>
                <table class="widefat">
                    <tr>
                        <td><strong><?php _e(
                            "Plugin Version",
                            "ahur-system",
                        ); ?></strong></td>
                        <td><?php echo esc_html(AHUR_SYSTEM_VERSION); ?></td>
                    </tr>
                    <tr>
                        <td><strong><?php _e(
                            "WordPress Version",
                            "ahur-system",
                        ); ?></strong></td>
                        <td><?php echo esc_html(
                            get_bloginfo("version"),
                        ); ?></td>
                    </tr>
                    <tr>
                        <td><strong><?php _e(
                            "PHP Version",
                            "ahur-system",
                        ); ?></strong></td>
                        <td><?php echo esc_html(PHP_VERSION); ?></td>
                    </tr>
                    <tr>
                        <td><strong><?php _e(
                            "API Status",
                            "ahur-system",
                        ); ?></strong></td>
                        <td id="ahur-api-status">
                            <?php
                            // Server-side API status check with detailed debugging
                            $api_url = get_option(
                                "ahur_system_api_url",
                                AHUR_SYSTEM_API_BASE,
                            );
                            $response = wp_remote_get(
                                $api_url . "/plugins/info?limit=1",
                                [
                                    "timeout" => 10,
                                    "headers" => [
                                        "User-Agent" =>
                                            "WordPress/" .
                                            get_bloginfo("version") .
                                            "; " .
                                            get_bloginfo("url"),
                                    ],
                                ],
                            );

                            if (
                                !is_wp_error($response) &&
                                wp_remote_retrieve_response_code($response) ===
                                    200
                            ) {
                                echo '<span style="color: green;">' .
                                    esc_html__("Connected", "ahur-system") .
                                    "</span>";
                            } else {
                                echo '<span style="color: red;">' .
                                    esc_html__("Disconnected", "ahur-system") .
                                    "</span>";

                                // Show detailed error for debugging
                                if (is_wp_error($response)) {
                                    echo "<br><small><code>" .
                                        esc_html(
                                            $response->get_error_message(),
                                        ) .
                                        "</code></small>";
                                } else {
                                    $code = wp_remote_retrieve_response_code(
                                        $response,
                                    );
                                    $message = wp_remote_retrieve_response_message(
                                        $response,
                                    );
                                    echo "<br><small><code>HTTP " .
                                        esc_html($code) .
                                        ": " .
                                        esc_html($message) .
                                        "</code></small>";
                                }
                            }
                            ?>
                        </td>
                    </tr>
                </table>
            </div>

            <div class="card">
                <h2><?php _e("Debug Information", "ahur-system"); ?></h2>
                <p><?php _e(
                    "This information helps troubleshoot update detection issues.",
                    "ahur-system",
                ); ?></p>

                <table class="widefat">
                    <tr>
                        <td><strong><?php _e(
                            "Installed Plugins",
                            "ahur-system",
                        ); ?></strong></td>
                        <td>
                            <?php
                            if (!function_exists("get_plugins")) {
                                require_once ABSPATH .
                                    "wp-admin/includes/plugin.php";
                            }
                            $all_plugins = get_plugins();
                            $ahur_plugins = [];
                            foreach (
                                $all_plugins
                                as $plugin_file => $plugin_data
                            ) {
                                if (
                                    strpos($plugin_file, "ahur") !== false ||
                                    strpos($plugin_file, "rss") !== false ||
                                    strpos($plugin_data["Name"], "Ahur") !==
                                        false ||
                                    strpos($plugin_data["Name"], "RSS") !==
                                        false
                                ) {
                                    $ahur_plugins[$plugin_file] =
                                        $plugin_data["Version"];
                                }
                            }

                            if (!empty($ahur_plugins)) {
                                echo "<ul>";
                                foreach (
                                    $ahur_plugins
                                    as $plugin_file => $version
                                ) {
                                    echo "<li><code>" .
                                        esc_html($plugin_file) .
                                        "</code> - v" .
                                        esc_html($version) .
                                        "</li>";

                                    // Also show extracted slug for debugging
                                    $slug = dirname($plugin_file);
                                    if ($slug !== ".") {
                                        echo "<br><small>Slug: <code>" .
                                            esc_html($slug) .
                                            "</code></small>";
                                    }
                                }
                                echo "</ul>";
                            } else {
                                echo "<em>" .
                                    esc_html__(
                                        "No Ahur plugins found",
                                        "ahur-system",
                                    ) .
                                    "</em>";
                            }
                            ?>
                        </td>
                    </tr>
                    <tr>
                        <td><strong><?php _e(
                            "WordPress Update Check",
                            "ahur-system",
                        ); ?></strong></td>
                        <td>
                            <?php
                            // Show what WordPress update system sees
                            if (!function_exists("get_site_transient")) {
                                require_once ABSPATH . "wp-includes/option.php";
                            }
                            $update_plugins = get_site_transient(
                                "update_plugins",
                            );
                            if (
                                $update_plugins &&
                                isset($update_plugins->checked)
                            ) {
                                echo "<h4>WordPress is checking these plugins:</h4>";
                                echo "<ul>";
                                foreach (
                                    $update_plugins->checked
                                    as $plugin_file => $version
                                ) {
                                    if (
                                        strpos($plugin_file, "ahur") !==
                                            false ||
                                        strpos($plugin_file, "rss") !== false
                                    ) {
                                        echo "<li><code>" .
                                            esc_html($plugin_file) .
                                            "</code> -> v" .
                                            esc_html($version) .
                                            "</li>";
                                    }
                                }
                                echo "</ul>";

                                if (
                                    isset($update_plugins->response) &&
                                    !empty($update_plugins->response)
                                ) {
                                    echo "<h4>Updates found:</h4>";
                                    echo "<ul>";
                                    foreach (
                                        $update_plugins->response
                                        as $plugin_file => $update
                                    ) {
                                        if (
                                            strpos($plugin_file, "ahur") !==
                                                false ||
                                            strpos($plugin_file, "rss") !==
                                                false
                                        ) {
                                            echo "<li><code>" .
                                                esc_html($plugin_file) .
                                                "</code> -> v" .
                                                esc_html($update->new_version) .
                                                "</li>";
                                        }
                                    }
                                    echo "</ul>";
                                } else {
                                    echo "<h4>No updates found</h4>";
                                }
                            } else {
                                echo "<em>Update check not run yet</em>";
                            }
                            ?>
                        </td>
                    </tr>
                    <tr>
                        <td><strong><?php _e(
                            "API Base URL",
                            "ahur-system",
                        ); ?></strong></td>
                        <td><code><?php echo esc_html(
                            get_option(
                                "ahur_system_api_url",
                                AHUR_SYSTEM_API_BASE,
                            ),
                        ); ?></code></td>
                    </tr>
                    <tr>
                        <td><strong><?php _e(
                            "Cache Time",
                            "ahur-system",
                        ); ?></strong></td>
                        <td><?php echo esc_html(
                            get_option("ahur_system_cache_time", 6),
                        ); ?> hours</td>
                    </tr>
                    <tr>
                        <td><strong><?php _e(
                            "Updates Enabled",
                            "ahur-system",
                        ); ?></strong></td>
                        <td><?php echo get_option(
                            "ahur_system_enable_updates",
                            1,
                        )
                            ? esc_html__("Yes", "ahur-system")
                            : esc_html__("No", "ahur-system"); ?></td>
                    </tr>
                </table>
            </div>

            <?php submit_button(); ?>

            <div class="card">
                <h2><?php _e("Debug Tools", "ahur-system"); ?></h2>
                <p><?php _e(
                    "Use these tools to troubleshoot update detection issues.",
                    "ahur-system",
                ); ?></p>

                <table class="form-table" role="presentation">
                    <tbody>
                        <tr>
                            <th scope="row">
                                <?php _e("Clear Cache", "ahur-system"); ?>
                            </th>
                            <td>
                                <button type="button" id="ahur-clear-cache" class="button button-secondary">
                                    <?php _e(
                                        "Clear All Caches",
                                        "ahur-system",
                                    ); ?>
                                </button>
                                <p class="description"><?php _e(
                                    "Clear plugin and theme update caches to force refresh.",
                                    "ahur-system",
                                ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <?php _e("Test Connection", "ahur-system"); ?>
                            </th>
                            <td>
                                <button type="button" id="ahur-test-connection" class="button button-secondary">
                                    <?php _e(
                                        "Test API Connection",
                                        "ahur-system",
                                    ); ?>
                                </button>
                                <p class="description"><?php _e(
                                    "Test direct connection to API endpoint.",
                                    "ahur-system",
                                ); ?></p>
                                <div id="ahur-connection-result" style="margin-top: 10px;"></div>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <?php _e(
                                    "Force Update Check",
                                    "ahur-system",
                                ); ?>
                            </th>
                            <td>
                                <button type="button" id="ahur-force-check" class="button button-secondary">
                                    <?php _e(
                                        "Check for Updates Now",
                                        "ahur-system",
                                    ); ?>
                                </button>
                                <p class="description"><?php _e(
                                    "Manually trigger update check for plugins and themes.",
                                    "ahur-system",
                                ); ?></p>
                            </td>
                        </tr>
                    </tbody>
                </table>
            </div>

            </form>
        </div>

        <script>
        jQuery(document).ready(function($) {
            // Check API status
            $.ajax({
                url: '<?php echo esc_js(
                    AHUR_SYSTEM_API_BASE,
                ); ?>/plugins/info?limit=1',
                method: 'GET',
                timeout: 5000,
                success: function(data) {
                    if (data && data.success) {
                        $('#ahur-api-status').html('<span style="color: green;"><?php echo esc_js(
                            __("Connected", "ahur-system"),
                        ); ?></span>');
                    } else {
                        $('#ahur-api-status').html('<span style="color: red;"><?php echo esc_js(
                            __("API Error", "ahur-system"),
                        ); ?></span>');
                    }
                },
                error: function(xhr, status, error) {
                    console.log('API Status Error:', status, error);
                    $('#ahur-api-status').html('<span style="color: red;"><?php echo esc_js(
                        __("Disconnected", "ahur-system"),
                    ); ?> (' + status + ')</span>');
                }
            });

            // Clear cache
            $('#ahur-clear-cache').on('click', function() {
                var $btn = $(this);
                $btn.prop('disabled', true).text('<?php echo esc_js(
                    __("Clearing...", "ahur-system"),
                ); ?>');

                $.post(ajaxurl, {
                    action: 'ahur_clear_cache',
                    _ajax_nonce: '<?php echo wp_create_nonce(
                        "ahur_clear_cache",
                    ); ?>'
                })
                .done(function(response) {
                    if (response.success) {
                        $btn.text('<?php echo esc_js(
                            __("Cache Cleared!", "ahur-system"),
                        ); ?>');
                        setTimeout(function() {
                            $btn.prop('disabled', false).text('<?php echo esc_js(
                                __("Clear All Caches", "ahur-system"),
                            ); ?>');
                        }, 2000);
                    } else {
                        $btn.prop('disabled', false).text('<?php echo esc_js(
                            __("Error", "ahur-system"),
                        ); ?>');
                    }
                })
                .fail(function() {
                    $btn.prop('disabled', false).text('<?php echo esc_js(
                        __("Error", "ahur-system"),
                    ); ?>');
                });
            });

            // Test connection
            $('#ahur-test-connection').on('click', function() {
                var $btn = $(this);
                var $result = $('#ahur-connection-result');

                $btn.prop('disabled', true).text('<?php echo esc_js(
                    __("Testing...", "ahur-system"),
                ); ?>');
                $result.html('<div style="color: #666;"><?php echo esc_js(
                    __("Testing connection to API...", "ahur-system"),
                ); ?></div>');

                $.post(ajaxurl, {
                    action: 'ahur_test_connection',
                    _ajax_nonce: '<?php echo wp_create_nonce(
                        "ahur_test_connection",
                    ); ?>'
                })
                .done(function(response) {
                    if (response.success) {
                        $result.html('<div style="color: green;"><strong><?php echo esc_js(
                            __("✓ Connection Successful!", "ahur-system"),
                        ); ?></strong><br>' +
                            '<?php echo esc_js(
                                __("Plugins found:", "ahur-system"),
                            ); ?> ' + response.data.plugins_found + '<br>' +
                            '<?php echo esc_js(
                                __("Response preview:", "ahur-system"),
                            ); ?> <code>' + response.data.response_preview + '...</code></div>');
                    } else {
                        $result.html('<div style="color: red;"><strong><?php echo esc_js(
                            __("✗ Connection Failed", "ahur-system"),
                        ); ?></strong><br>' +
                            '<?php echo esc_js(
                                __("Error:", "ahur-system"),
                            ); ?> ' + response.data.message + '<br>' +
                            '<?php echo esc_js(
                                __("Error Type:", "ahur-system"),
                            ); ?> ' + response.data.type + '</div>');
                    }
                    $btn.prop('disabled', false).text('<?php echo esc_js(
                        __("Test API Connection", "ahur-system"),
                    ); ?>');
                })
                .fail(function(xhr, status, error) {
                    $result.html('<div style="color: red;"><strong><?php echo esc_js(
                        __("✗ AJAX Request Failed", "ahur-system"),
                    ); ?></strong><br>' +
                        '<?php echo esc_js(
                            __("Status:", "ahur-system"),
                        ); ?> ' + status + '<br>' +
                        '<?php echo esc_js(
                            __("Error:", "ahur-system"),
                        ); ?> ' + error + '</div>');
                    $btn.prop('disabled', false).text('<?php echo esc_js(
                        __("Test API Connection", "ahur-system"),
                    ); ?>');
                });
            });

            // Force update check
            $('#ahur-force-check').on('click', function() {
                var $btn = $(this);
                $btn.prop('disabled', true).text('<?php echo esc_js(
                    __("Checking...", "ahur-system"),
                ); ?>');

                $.post(ajaxurl, {
                    action: 'ahur_force_check',
                    _ajax_nonce: '<?php echo wp_create_nonce(
                        "ahur_force_check",
                    ); ?>'
                })
                .done(function(response) {
                    if (response.success) {
                        $btn.text('<?php echo esc_js(
                            __("Update Check Complete!", "ahur-system"),
                        ); ?>');
                        // Redirect to updates page
                        setTimeout(function() {
                            window.location.href = '<?php echo admin_url(
                                "update-core.php",
                            ); ?>';
                        }, 1500);
                    } else {
                        $btn.prop('disabled', false).text('<?php echo esc_js(
                            __("Error", "ahur-system"),
                        ); ?>');
                    }
                })
                .fail(function() {
                    $btn.prop('disabled', false).text('<?php echo esc_js(
                        __("Error", "ahur-system"),
                    ); ?>');
                });
            });
        });
        </script>

        <script>
        jQuery(document).ready(function($) {
            // Check API status
            $.ajax({
                url: '<?php echo esc_js(
                    AHUR_SYSTEM_API_BASE,
                ); ?>/plugins/info?limit=1',
                method: 'GET',
                timeout: 5000,
                success: function(data) {
                    if (data && data.success) {
                        $('#ahur-api-status').html('<span style="color: green;"><?php echo esc_js(
                            __("Connected", "ahur-system"),
                        ); ?></span>');
                    } else {
                        $('#ahur-api-status').html('<span style="color: red;"><?php echo esc_js(
                            __("API Error", "ahur-system"),
                        ); ?></span>');
                    }
                },
                error: function(xhr, status, error) {
                    console.log('API Status Error:', status, error);
                    $('#ahur-api-status').html('<span style="color: red;"><?php echo esc_js(
                        __("Disconnected", "ahur-system"),
                    ); ?> (' + status + ')</span>');
                }
            });
        });
        </script>
        <?php
    }

    /**
     * Clear all caches
     */
    private function clearAllCaches()
    {
        global $wpdb;

        // Clear our transients
        $wpdb->query(
            "DELETE FROM {$wpdb->options}
             WHERE option_name LIKE '_transient_custom_repo_%'",
        );

        // Clear theme repository transients
        $wpdb->query(
            "DELETE FROM {$wpdb->options}
             WHERE option_name LIKE '_transient_ahur_themes_list_%'",
        );

        $theme_cache_keys = $wpdb->get_col(
            "SELECT option_name FROM {$wpdb->options} WHERE option_name LIKE '_transient_ahur_theme_%'"
        );

        foreach ($theme_cache_keys as $cache_key) {
            delete_transient(str_replace('_transient_', '', $cache_key));
        }

        // Clear plugin caches
        if (function_exists("wp_cache_flush")) {
            wp_cache_flush();
        }
    }

    /**
     * AJAX handler for clearing cache
     */
    public function ajaxClearCache()
    {
        check_ajax_referer("ahur_clear_cache", "_ajax_nonce");

        if (!current_user_can("manage_options")) {
            wp_die(__("Unauthorized", "ahur-system"));
        }

        $this->clearAllCaches();

        // Clear WordPress update caches
        wp_clean_plugins_cache();
        wp_cache_delete("update_plugins", "update_plugins");
        wp_cache_delete("update_themes", "update_themes");

        wp_send_json_success();
    }

    /**
     * AJAX handler for forcing update check
     */
    public function ajaxForceCheck()
    {
        check_ajax_referer("ahur_force_check", "_ajax_nonce");

        if (!current_user_can("manage_options")) {
            wp_die(__("Unauthorized", "ahur-system"));
        }

        // Clear caches first
        $this->clearAllCaches();
        wp_clean_plugins_cache();
        wp_cache_delete("update_plugins", "update_plugins");
        wp_cache_delete("update_themes", "update_themes");

        // Force WordPress to check for updates
        wp_version_check();
        wp_update_plugins();
        wp_update_themes();

        wp_send_json_success();
    }

    /**
     * AJAX handler for testing API connection
     */
    public function ajaxTestConnection()
    {
        check_ajax_referer("ahur_test_connection", "_ajax_nonce");

        if (!current_user_can("manage_options")) {
            wp_die(__("Unauthorized", "ahur-system"));
        }

        $api_url = get_option("ahur_system_api_url", AHUR_SYSTEM_API_BASE);

        // Test basic connectivity
        $response = wp_remote_get($api_url . "/plugins/info?limit=1", [
            "timeout" => 15,
            "headers" => [
                "User-Agent" =>
                    "WordPress/" .
                    get_bloginfo("version") .
                    "; " .
                    get_bloginfo("url"),
            ],
        ]);

        if (is_wp_error($response)) {
            wp_send_json_error([
                "message" => $response->get_error_message(),
                "type" => "wp_error",
            ]);
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($code !== 200) {
            wp_send_json_error([
                "message" =>
                    "HTTP {$code}: " .
                    wp_remote_retrieve_response_message($response),
                "type" => "http_error",
                "code" => $code,
            ]);
        }

        $data = json_decode($body);

        if (!$data || !isset($data->success)) {
            wp_send_json_error([
                "message" => "Invalid JSON response",
                "type" => "json_error",
                "body" => substr($body, 0, 200),
            ]);
        }

        wp_send_json_success([
            "message" => "Connection successful",
            "plugins_found" => isset($data->data) ? count($data->data) : 0,
            "response_preview" => substr($body, 0, 300),
        ]);
    }

    /**
     * AJAX handler for checking plugin status
     */
    public function ajaxCheckPluginStatus()
    {
        check_ajax_referer("ahur_check_plugin_status", "_ajax_nonce");

        if (!current_user_can("manage_options")) {
            wp_die(__("Unauthorized", "ahur-system"));
        }

        $slug = sanitize_text_field($_POST["slug"]);

        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $all_plugins = get_plugins();
        $is_installed = false;
        $installed_version = '0.0.0';
        $update_available = false;

        foreach ($all_plugins as $plugin_file => $plugin_data) {
            $plugin_slug = dirname($plugin_file);
            $plugin_slug = $plugin_slug === '.' ? basename($plugin_file, '.php') : $plugin_slug;
            $plugin_slug_lower = strtolower($plugin_slug);
            $request_slug_lower = strtolower($slug);

            // Try multiple matching strategies (same as other methods)
            if ($plugin_slug_lower === $request_slug_lower ||
                str_replace(['-', '_'], '', $plugin_slug_lower) === str_replace(['-', '_'], '', $request_slug_lower) ||
                strpos($plugin_slug_lower, $request_slug_lower) !== false ||
                strpos($request_slug_lower, $plugin_slug_lower) !== false) {
                $is_installed = true;
                $installed_version = $plugin_data['Version'];
                error_log("Ahur System: AJAX - Plugin {$slug} is installed as {$plugin_file}");
                break;
            }
        }

        if ($is_installed) {
            // Check for updates from our repository
            $api_url = get_option("ahur_system_api_url", AHUR_SYSTEM_API_BASE);
            $response = wp_remote_get($api_url . '/plugins/info?query=' . urlencode($slug), [
                "timeout" => 10,
                "headers" => [
                    "User-Agent" => "WordPress/" . get_bloginfo("version") . "; " . get_bloginfo("url"),
                ],
            ]);

            if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
                $body = wp_remote_retrieve_body($response);
                $data = json_decode($body);

                if ($data && isset($data->data) && !empty($data->data)) {
                    $remote_plugin = $data->data[0];
                    $remote_version = $remote_plugin->version ?? '0.0.0';
                    $update_available = version_compare($remote_version, $installed_version, '>');
                }
            }
        }

        wp_send_json_success([
            "installed" => $is_installed,
            "installed_version" => $installed_version,
            "update_available" => $update_available,
            "update_url" => admin_url('update-core.php'),
        ]);
    }

    /**
     * Add JavaScript to handle plugin activation retry
     */
    public function addActivationRetryScript()
    {
        // Only add this script on plugin installation pages
        $screen = get_current_screen();
        if (
            !$screen ||
            !in_array($screen->id, ["plugin-install", "update-core"])
        ) {
            return;
        }?>
        <script>
        jQuery(document).ready(function($) {
            // Override the original plugin activation handler to add retry logic
            var originalActivatePlugin = window.wp.updates.activatePlugin;

            if (window.wp && window.wp.updates && originalActivatePlugin) {
                window.wp.updates.activatePlugin = function(args) {
                    var self = this;
                    var originalSuccess = args.success;
                    var originalError = args.error;
                    var retryCount = 0;
                    var maxRetries = 3;

                    function attemptActivation() {
                        args.success = function(response) {
                            if (originalSuccess) {
                                originalSuccess.call(self, response);
                            }
                        };

                        args.error = function(response) {
                            retryCount++;

                            // If it's a "plugin_not_installed" error and we haven't exceeded retries, try again
                            if (response.data && response.data.errorCode === 'plugin_not_installed' && retryCount <= maxRetries) {
                                console.log('Ahur System: Plugin activation failed, retrying... (' + retryCount + '/' + maxRetries + ')');

                                setTimeout(function() {
                                    originalActivatePlugin.call(self, args);
                                }, 2000 * retryCount); // Exponential backoff
                                return;
                            }

                            // If we've exhausted retries or it's a different error, call original error handler
                            if (originalError) {
                                originalError.call(self, response);
                            }
                        };

                        originalActivatePlugin.call(self, args);
                    }

                    attemptActivation();
                };
            }
        });
        </script>
        <?php
    }

    /**
     * Debug plugin slugs for troubleshooting
     */
    public function debugPluginSlugs()
    {
        if (isset($_GET['ahur_debug']) && current_user_can('manage_options')) {
            if (!function_exists('get_plugins')) {
                require_once ABSPATH . 'wp-admin/includes/plugin.php';
            }

            $all_plugins = get_plugins();
            echo '<h2>Installed Plugins Debug Info</h2>';
            echo '<table border="1" cellpadding="5" cellspacing="0">';
            echo '<tr><th>Plugin File</th><th>Plugin Name</th><th>Version</th><th>Directory Slug</th><th>Basename Slug</th></tr>';

            foreach ($all_plugins as $plugin_file => $plugin_data) {
                $dir_slug = dirname($plugin_file);
                $basename_slug = basename($plugin_file, '.php');
                echo '<tr>';
                echo '<td>' . esc_html($plugin_file) . '</td>';
                echo '<td>' . esc_html($plugin_data['Name']) . '</td>';
                echo '<td>' . esc_html($plugin_data['Version']) . '</td>';
                echo '<td>' . esc_html($dir_slug) . '</td>';
                echo '<td>' . esc_html($basename_slug) . '</td>';
                echo '</tr>';
            }
            echo '</table>';
            echo '<p><strong>Repository Plugins:</strong></p>';
            echo '<ul>';

            // Check repository plugins
            $api_url = get_option("ahur_system_api_url", AHUR_SYSTEM_API_BASE);
            $response = wp_remote_get($api_url . '/plugins/info', [
                "timeout" => 10,
                "headers" => [
                    "User-Agent" => "WordPress/" . get_bloginfo("version") . "; " . get_bloginfo("url"),
                ],
            ]);

            if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
                $body = wp_remote_retrieve_body($response);
                $data = json_decode($body);

                if ($data && isset($data->data)) {
                    foreach ($data->data as $plugin) {
                        echo '<li>' . esc_html($plugin->name) . ' (slug: ' . esc_html($plugin->slug) . ', version: ' . esc_html($plugin->version) . ')</li>';
                    }
                }
            }
            echo '</ul>';
            exit;
        }
    }

    /**
     * Add custom CSS for plugin installation page
     */
    public function addPluginInstallStyles()
    {
        // Only add this on plugin install page
        $screen = get_current_screen();
        if (!$screen || $screen->id !== "plugin-install") {
            return;
        }
        ?>
        <style>
        .plugin-card.custom-repository-plugin {
            border-left: 4px solid #0073aa;
        }
        .plugin-card.custom-repository-plugin .plugin-action-buttons .button {
            margin-right: 5px;
        }
        .plugin-card[data-source="ahur_repository"] .install-now {
            display: none;
        }
        .plugin-card[data-source="ahur_repository"][data-installed="true"] .install-now {
            display: none;
        }
        .plugin-card[data-source="ahur_repository"][data-installed="true"] .activate-now {
            display: none;
        }
        </style>
        <?php
    }

    /**
     * Add JavaScript to handle plugin install page button states
     */
    public function addPluginInstallPageScript()
    {
        // Only add this script on plugin install page
        $screen = get_current_screen();
        if (!$screen || $screen->id !== "plugin-install") {
            return;
        }
        ?>
        <script>
        jQuery(document).ready(function($) {
            // Update button states for custom repository plugins
            function updateCustomPluginButtons() {
                $('.plugin-card').each(function() {
                    var $card = $(this);
                    var $actions = $card.find('.plugin-action-buttons');
                    var slug = $card.data('slug');

                    // Check if this is a custom repository plugin
                    if ($card.find('[data-source="ahur_repository"]').length > 0 || $card.data('source') === 'ahur_repository') {
                        console.log('Ahur System: Found custom plugin card for slug:', slug);

                        // Check if plugin is installed by making AJAX call
                        $.post(ajaxurl, {
                            action: 'ahur_check_plugin_status',
                            slug: slug,
                            _ajax_nonce: '<?php echo wp_create_nonce("ahur_check_plugin_status"); ?>'
                        })
                        .done(function(response) {
                            if (response.success) {
                                if (response.data.installed) {
                                    // Plugin is installed
                                    if (response.data.update_available) {
                                        // Show update button
                                        $actions.html('<a class="button button-primary update-now" href="' + response.data.update_url + '"><?php echo esc_js(__("Update Now", "ahur-system")); ?></a>');
                                    } else {
                                        // Show active button
                                        $actions.html('<button class="button button-disabled" disabled><?php echo esc_js(__("Active", "ahur-system")); ?></button>');
                                    }
                                    $card.attr('data-installed', 'true');
                                } else {
                                    // Show install button
                                    $card.attr('data-installed', 'false');
                                }
                            }
                        })
                        .fail(function() {
                            console.log('Ahur System: Failed to check plugin status for:', slug);
                        });

                        // Add visual indicator for custom repository plugins
                        $card.addClass('custom-repository-plugin');
                    }
                });
            }

            // Run on page load
            updateCustomPluginButtons();

            // Run after AJAX updates (like search filtering)
            $(document).on('wp-updates-success wp-updates-error', function() {
                setTimeout(updateCustomPluginButtons, 1000);
            });

            // Handle installation errors for custom plugins
            $(document).on('wp-updates-error', function(e, response) {
                if (response && response.error && response.error.indexOf('پوشهٔ هدف از قبل موجود است') !== -1) {
                    // Directory exists error - this means the plugin is actually installed
                    var $card = $('.plugin-card').filter(function() {
                        return $(this).find('.install-now').data('slug') === response.slug;
                    });

                    if ($card.length) {
                        // Force refresh the page to show correct button state
                        setTimeout(function() {
                            window.location.reload();
                        }, 2000);

                        // Show user-friendly message
                        $card.find('.plugin-action-buttons').html(
                            '<div class="notice notice-warning inline"><p>' +
                            '<?php echo esc_js(__("Plugin appears to be installed. Refreshing...", "ahur-system")); ?>' +
                            '</p></div>'
                        );
                    }
                }
            });
        });
        </script>
        <?php
    }
}

// Initialize the plugin
AhurSystem::getInstance();
