<?php
/**
 * Plugin Repository Manager
 *
 * Handles all plugin-related operations for the Ahur repository
 *
 * @package AhurSystem
 * @author Ahur Team
 * @version 0.0.9
 */

// Prevent direct access
if (!defined("ABSPATH")) {
    exit();
}

class AhurPluginRepository
{
    /**
     * API base URL
     */
    private $api_base;

    /**
     * Cache time in hours
     */
    private $cache_time;

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->api_base = get_option(
            "ahur_system_api_url",
            AHUR_SYSTEM_API_BASE,
        );
        $this->cache_time = (int) get_option("ahur_system_cache_time", 6);
    }

    /**
     * Initialize plugin hooks
     */
    public function initHooks()
    {

        // Plugin updates and info
        add_filter("pre_set_site_transient_update_plugins", [$this, "checkPluginUpdates"]);
        add_filter("plugins_api", [$this, "getPluginInfo"], 5, 3); // Even higher priority
        add_filter("plugins_api_result", [$this, "injectPlugins"], 999, 3);
        
        // Force WordPress to use our repository for plugin info
        add_filter("site_transient_update_plugins", [$this, "forcePluginInfoCheck"], 10, 2);

        // Installation handling
        add_filter("upgrader_source_selection", [$this, "handleInstallation"], 10, 4);
        add_filter("upgrader_pre_download", [$this, "preDownload"], 10, 3);
        
        // Handle installation errors and directory conflicts
        add_filter("upgrader_post_install", [$this, "postInstall"], 10, 3);

        // Cache clearing
        add_action("upgrader_process_complete", [$this, "clearCache"], 10, 2);
        add_action("activated_plugin", [$this, "clearCache"]);
        add_action("deactivated_plugin", [$this, "clearCache"]);
        
        // Clear plugin info cache when viewing details - run on init to avoid early loading
        add_action("init", [$this, "clearPluginInfoCache"]);

        // Force refresh plugin status on Ahur tab - run on init to avoid early loading
        add_action("init", [$this, "forceRefreshPluginStatus"]);

        // Custom Ahur tab for plugin installation
        add_filter("install_plugins_tabs", [$this, "addAhurTab"]);
        add_filter("install_plugins_table_api_args_ahur", [$this, "getAhurTabArgs"]);
        add_action("install_plugins_ahur", [$this, "displayAhurTabContent"]);
        add_filter("plugins_api", [$this, "handleAhurTabApi"], 10, 3);
        
        // Fix button states for installed plugins
        add_filter("plugin_install_action_links", [$this, "fixPluginActionLinks"], 10, 2);
        
        // Modify plugin data after API response to set correct installation status
        add_filter("plugins_api_result", [$this, "modifyPluginInstallStatus"], 10, 3);
    }

    /**
     * Fetch plugins list from API with pagination support
     */
    public function fetchPluginsList($page = 1, $per_page = 36)
    {
        $url = $this->api_base . '/plugins/info';
        $params = [];
        
        // Add pagination parameters
        if ($page > 1) {
            $params['page'] = $page;
        }
        if ($per_page !== 36) {
            $params['per_page'] = $per_page;
        }
        
        if (!empty($params)) {
            $url .= '?' . http_build_query($params);
        }
        
        $cache_key = "ahur_plugins_list_" . md5($url);

        // Check cache first
        $cached = get_transient($cache_key);
        if ($cached !== false) {
            return $cached;
        }

        $response = wp_remote_get($url, [
            "timeout" => 10,
            "headers" => [
                "User-Agent" => "WordPress/" . get_bloginfo("version") . "; " . get_bloginfo("url"),
            ],
        ]);

        if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
            error_log('Ahur System: API request failed: ' . (is_wp_error($response) ? $response->get_error_message() : 'HTTP ' . wp_remote_retrieve_response_code($response)));
            return false;
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body);

        if (!$data) {
            error_log('Ahur System: Failed to decode JSON response');
            return false;
        }

        // Cache the full response for 1 hour
        set_transient($cache_key, $data, HOUR_IN_SECONDS);

        return $data;
    }

    /**
     * Search plugins in repository
     */
    public function searchPlugins($search_term, $page = 1, $per_page = 36)
    {
        $url = $this->api_base . '/plugins/info';
        $params = [
            'search' => $search_term,
            'format' => 'html'
        ];
        
        // Add pagination parameters
        if ($page > 1) {
            $params['page'] = $page;
        }
        if ($per_page !== 36) {
            $params['per_page'] = $per_page;
        }
        
        $url .= '?' . http_build_query($params);
        $cache_key = "ahur_plugins_search_" . md5($url);

        // Check cache first
        $cached = get_transient($cache_key);
        if ($cached !== false) {
            return $cached;
        }

        $response = wp_remote_get($url, [
            "timeout" => 10,
            "headers" => [
                "User-Agent" => "WordPress/" . get_bloginfo("version") . "; " . get_bloginfo("url"),
            ],
        ]);

        if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
            error_log('Ahur System: API search request failed: ' . (is_wp_error($response) ? $response->get_error_message() : 'HTTP ' . wp_remote_retrieve_response_code($response)));
            return false;
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body);

        if (!$data) {
            error_log('Ahur System: Failed to decode JSON response');
            return false;
        }

        // Cache the search results for 30 minutes
        set_transient($cache_key, $data, 30 * MINUTE_IN_SECONDS);

        return $data;
    }

    /**
     * Unified method to fetch data from API with caching
     */
    private function fetchFromAPI($endpoint, $slug = null, $useCache = true)
    {
        $url = $this->api_base . $endpoint;
        if ($slug) {
            // Always use lowercase slug for consistency
            $slug = strtolower($slug);
            $url .= "?query=" . urlencode($slug);
        }

        $cache_key = "ahur_" . md5($url);

        if ($useCache) {
            $cached = get_transient($cache_key);
            if ($cached !== false) {
                return $cached;
            }
        }

        $response = wp_remote_get($url, [
            "timeout" => 10,
            "headers" => [
                "User-Agent" => "WordPress/" . get_bloginfo("version") . "; " . get_bloginfo("url"),
            ],
        ]);

        if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
            return false;
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body);

        if (!$data || !isset($data->data)) {
            return false;
        }

        $result = is_array($data->data) ? $data->data : [$data->data];

        if ($useCache) {
            set_transient($cache_key, $result, $this->cache_time * HOUR_IN_SECONDS);
        }

        return $result;
    }

    /**
     * Smart sanitization - convert objects to proper WordPress format
     */
    private function sanitizeData($data)
    {
        if (!is_object($data)) {
            return $data;
        }

        $sanitized = new stdClass();

        foreach (get_object_vars($data) as $key => $value) {
            if (is_array($value)) {
                // Convert arrays to strings or skip
                if ($key === 'tags' || $key === 'versions') {
                    $sanitized->$key = implode(', ', $value);
                } elseif ($key === 'icons' || $key === 'banners') {
                    $sanitized->$key = $this->sanitizeBanners($value);
                } else {
                    $sanitized->$key = ''; // Skip other arrays
                }
            } elseif (is_object($value)) {
                // Handle nested objects (like sections)
                if ($key === 'sections') {
                    $sanitized->$key = $this->sanitizeSections($value);
                } else {
                    $sanitized->$key = ''; // Skip other objects
                }
            } else {
                $sanitized->$key = $value;
            }
        }

        return $sanitized;
    }

    /**
     * Sanitize sections object to array
     */
    private function sanitizeSections($sections)
    {
        if (!is_object($sections)) {
            return $sections;
        }

        $sanitized = [];
        foreach (get_object_vars($sections) as $key => $value) {
            if (!is_array($value) && !is_object($value)) {
                $sanitized[$key] = $value;
            } else {
                $sanitized[$key] = ''; // Convert complex objects to empty strings
            }
        }

        return $sanitized;
    }

    /**
     * Sanitize banners/icons array
     */
    private function sanitizeBanners($banners)
    {
        if (!is_array($banners)) {
            return $banners;
        }

        $sanitized = [];
        foreach ($banners as $key => $value) {
            if (!is_array($value) && !is_object($value)) {
                $sanitized[$key] = $value;
            }
        }

        return $sanitized;
    }

    /**
     * Check if plugin is installed
     */
    private function isPluginInstalled($slug)
    {
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $all_plugins = get_plugins();
        $slug = strtolower($slug);
        
        foreach ($all_plugins as $plugin_file => $plugin_data) {
            $plugin_slug = strtolower(dirname($plugin_file));
            $plugin_slug = $plugin_slug === '.' ? strtolower(basename($plugin_file, '.php')) : $plugin_slug;
            
            // Try multiple matching strategies
            if ($plugin_slug === $slug ||
                str_replace(['-', '_'], '', $plugin_slug) === str_replace(['-', '_'], '', $slug) ||
                strpos($plugin_slug, $slug) !== false ||
                strpos($slug, $plugin_slug) !== false) {
                error_log("Ahur System: Plugin {$slug} is installed as {$plugin_file}");
                return true;
            }
        }
        
        error_log("Ahur System: Plugin {$slug} is not installed");
        return false;
    }

    /**
     * Check if slugs are similar (case-insensitive)
     */
    private function slugsSimilar($slug1, $slug2)
    {
        return strtolower($slug1) === strtolower($slug2);
    }

    /**
     * Test slug matching for debugging
     */
    public function testSlugMatching($repo_slug, $installed_plugins = null)
    {
        if ($installed_plugins === null) {
            if (!function_exists('get_plugins')) {
                require_once ABSPATH . 'wp-admin/includes/plugin.php';
            }
            $installed_plugins = get_plugins();
        }

        $repo_slug_lower = strtolower($repo_slug);
        $matches = [];

        foreach ($installed_plugins as $plugin_file => $plugin_data) {
            $plugin_slug = dirname($plugin_file);
            $plugin_slug = $plugin_slug === '.' ? basename($plugin_file, '.php') : $plugin_slug;
            $plugin_slug_lower = strtolower($plugin_slug);

            $is_match = ($plugin_slug_lower === $repo_slug_lower ||
                        str_replace(['-', '_'], '', $plugin_slug_lower) === str_replace(['-', '_'], '', $repo_slug_lower) ||
                        strpos($plugin_slug_lower, $repo_slug_lower) !== false ||
                        strpos($repo_slug_lower, $plugin_slug_lower) !== false);

            if ($is_match) {
                $matches[] = [
                    'file' => $plugin_file,
                    'slug' => $plugin_slug,
                    'name' => $plugin_data['Name'],
                    'version' => $plugin_data['Version']
                ];
            }
        }

        return $matches;
    }

    /**
     * Check for plugin updates
     */
    public function checkPluginUpdates($transient)
    {
        if (empty($transient->checked)) {
            return $transient;
        }

        $response = $this->fetchPluginsList();
        
        if (!$response || !$response->success || !isset($response->data)) {
            return $transient;
        }

        $plugins = $response->data;

        foreach ($plugins as $plugin) {
            $plugin_slug = $plugin->slug;
            
            // Debug: Log plugin data structure
            if (isset($plugin->icons)) {
                error_log("Ahur System: Plugin {$plugin_slug} has icons: " . print_r($plugin->icons, true));
            }

            // Normalize remote slug to lowercase
            $remote_slug = strtolower($plugin_slug);
            
            // Try to match plugin by slug with more flexible matching
            $matched_plugin_key = null;
            foreach ($transient->checked as $installed_plugin_key => $installed_version) {
                $installed_slug = strtolower(dirname($installed_plugin_key));
                $installed_slug = $installed_slug === '.' ? basename($installed_plugin_key, '.php') : $installed_slug;

                // Direct slug match
                if ($installed_slug === $remote_slug) {
                    $matched_plugin_key = $installed_plugin_key;
                    error_log("Ahur System: Direct slug match: " . $installed_plugin_key);
                    break;
                }
                
                // Try matching the full plugin file name
                if (strtolower($installed_plugin_key) === $remote_slug . '.php') {
                    $matched_plugin_key = $installed_plugin_key;
                    error_log("Ahur System: Full filename match: " . $installed_plugin_key);
                    break;
                }
                
                // Try matching with hyphen/underscore variations
                $installed_clean = str_replace(['-', '_'], '', $installed_slug);
                $remote_clean = str_replace(['-', '_'], '', $remote_slug);
                if ($installed_clean === $remote_clean) {
                    $matched_plugin_key = $installed_plugin_key;
                    error_log("Ahur System: Clean slug match: " . $installed_plugin_key);
                    break;
                }
                
                // Try matching with common prefixes
                if (strpos($installed_slug, $remote_slug) !== false || strpos($remote_slug, $installed_slug) !== false) {
                    $matched_plugin_key = $installed_plugin_key;
                    error_log("Ahur System: Partial match: " . $installed_plugin_key);
                    break;
                }
            }

            if ($matched_plugin_key) {
                $installed_version = $transient->checked[$matched_plugin_key];
                $remote_version = $plugin->version ?? "unknown";

                if (version_compare($remote_version, $installed_version, ">")) {
                    $update = new stdClass();
                    $update->slug = $plugin_slug;
                    $update->plugin = $matched_plugin_key;
                    $update->new_version = $remote_version;
                    $update->package = $plugin->downloadlink ?? '';
                    $update->url = $plugin->author_url ?? '';
                    $update->tested = $plugin->tested ?? "";
                    $update->requires = $plugin->requires ?? "";
                    $update->requires_php = $plugin->requires_php ?? "";

                    // Add icons if available
                    if (isset($plugin->icons) && is_object($plugin->icons)) {
                        $icons_array = [];
                        if (isset($plugin->icons->default)) {
                            $icons_array['1x'] = $plugin->icons->default;
                            $icons_array['2x'] = $plugin->icons->default;
                        }
                        foreach ($plugin->icons as $key => $url) {
                            if ($key !== 'default' && is_string($url)) {
                                $icons_array[$key] = $url;
                            }
                        }
                        if (!empty($icons_array)) {
                            $update->icons = $icons_array;
                        }
                    }

                    $transient->response[$matched_plugin_key] = $update;
                    
                    // Debug logging
                    error_log("Ahur System: Update found for {$matched_plugin_key}: {$installed_version} -> {$remote_version}");
                    if (isset($update->icons)) {
                        error_log("Ahur System: Icons added to update: " . print_r($update->icons, true));
                    }
                }
            }
        }

        return $transient;
    }

    /**
     * Get plugin information
     */
    public function getPluginInfo($false, $action, $response)
    {
        if ($action !== "plugin_information") {
            return $false;
        }

        $plugin_slug = $response->slug;
        
        // Debug logging
        error_log("Ahur System: Looking for plugin info for slug: " . $plugin_slug);
        
        // Check if this might be an installed plugin with different file name
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        
        $all_plugins = get_plugins();
        $actual_plugin_file = null;
        
        // Normalize the requested slug
        $normalized_slug = strtolower($plugin_slug);
        
        // Try to find the actual plugin file
        foreach ($all_plugins as $plugin_file => $plugin_data) {
            $file_slug = dirname($plugin_file);
            $file_slug = $file_slug === '.' ? basename($plugin_file, '.php') : $file_slug;
            $file_slug = strtolower($file_slug);
            
            if ($file_slug === $normalized_slug ||
                str_replace(['-', '_'], '', $file_slug) === str_replace(['-', '_'], '', $normalized_slug)) {
                $actual_plugin_file = $plugin_file;
                error_log("Ahur System: Found actual plugin file: " . $actual_plugin_file . " for slug: " . $plugin_slug);
                break;
            }
        }
        
        // Always normalize slug to lowercase
        $plugin_slug = strtolower($plugin_slug);
        
        error_log("Ahur System: Looking for plugin info for normalized slug: " . $plugin_slug);
        
        // Try different slug variations (all lowercase)
        $slug_variations = [
            $plugin_slug,
            str_replace('-', '', $plugin_slug),
            str_replace(['-', '_'], '', $plugin_slug),
        ];
        
        $plugin = null;
        foreach ($slug_variations as $slug) {
            error_log("Ahur System: Trying slug variation: " . $slug);
            $search_result = $this->searchPlugins($slug, 1, 10);
            
            if ($search_result && $search_result->success && !empty($search_result->data)) {
                $plugin = $search_result->data[0];
                error_log("Ahur System: Found plugin with slug: " . $slug);
                break;
            }
        }

        // Fallback: search all plugins if direct match fails
        if (!$plugin) {
            error_log("Ahur System: Direct slug match failed, searching all plugins...");
            $all_plugins_response = $this->fetchPluginsList();
            
            if ($all_plugins_response && $all_plugins_response->success && isset($all_plugins_response->data)) {
                foreach ($all_plugins_response->data as $repo_plugin) {
                    $repo_slug = strtolower($repo_plugin->slug);
                    
                    // Try various matching strategies (all lowercase)
                    if ($repo_slug === $plugin_slug ||
                        str_replace(['-', '_'], '', $repo_slug) === str_replace(['-', '_'], '', $plugin_slug) ||
                        strpos($repo_slug, $plugin_slug) !== false ||
                        strpos($plugin_slug, $repo_slug) !== false) {
                        $plugin = $repo_plugin;
                        error_log("Ahur System: Found plugin via fallback search: " . $repo_plugin->name . " (slug: " . $repo_plugin->slug . ")");
                        break;
                    }
                }
            }
        }

        if (!$plugin) {
            error_log("Ahur System: No plugins found for any slug variation of: " . $plugin_slug);
            return $false;
        }

        // Debug logging
        error_log("Ahur System: Found plugin: " . ($plugin->name ?? 'Unknown'));
        
        // Don't sanitize plugin data for getPluginInfo - we want to preserve all rich content
        // Create a proper response object
        $info = new stdClass();
        
        // Map plugin data to WordPress format
        $info->name = $plugin->name ?? "Unknown Plugin";
        $info->slug = $plugin->slug ?? $plugin_slug;
        $info->version = $plugin->version ?? "1.0.0";
        $info->author = $plugin->author ?? "";
        $info->homepage = $plugin->author_url ?? $plugin->homepage ?? "";
        $info->requires = $plugin->requires ?? "";
        $info->tested = $plugin->tested ?? "";
        $info->requires_php = $plugin->requires_php ?? "";
        $info->download_link = $plugin->downloadlink ?? $plugin->download_url ?? "";
        $info->trunk = $plugin->downloadlink ?? $plugin->download_url ?? "";

        // Handle sections - ensure all values are strings
        $sections = [];
        if (isset($plugin->sections) && is_object($plugin->sections)) {
            foreach (get_object_vars($plugin->sections) as $key => $value) {
                if (!empty($value)) {
                    // Convert to string if it's not already
                    if (is_string($value)) {
                        $sections[$key] = $value;
                    } elseif (is_array($value) || is_object($value)) {
                        // Handle arrays of objects (like screenshots)
                        if (is_array($value)) {
                            $array_strings = [];
                            foreach ($value as $item) {
                                if (is_object($item) && isset($item->url)) {
                                    // Create proper HTML image tag for screenshots
                                    $width = isset($item->width) ? $item->width : 800;
                                    $height = isset($item->height) ? $item->height : 600;
                                    $caption = isset($item->caption) && !empty($item->caption) ? esc_html($item->caption) : '';
                                    $array_strings[] = sprintf(
                                        '<img src="%s" width="%d" height="%d" alt="%s" style="max-width: 100%%; height: auto;" />',
                                        esc_url($item->url),
                                        intval($width),
                                        intval($height),
                                        $caption
                                    );
                                } elseif (is_object($item)) {
                                    $array_strings[] = print_r($item, true);
                                } else {
                                    $array_strings[] = (string) $item;
                                }
                            }
                            $sections[$key] = implode("\n\n", $array_strings);
                        } else {
                            $sections[$key] = json_encode($value);
                        }
                    } else {
                        $sections[$key] = (string) $value;
                    }
                }
            }
        } else {
            // Fallback to individual properties
            if (isset($plugin->description) && !empty($plugin->description)) {
                $sections['description'] = is_string($plugin->description) ? $plugin->description : (string) $plugin->description;
            }
            if (isset($plugin->installation) && !empty($plugin->installation)) {
                $sections['installation'] = is_string($plugin->installation) ? $plugin->installation : (string) $plugin->installation;
            }
            if (isset($plugin->changelog) && !empty($plugin->changelog)) {
                $sections['changelog'] = is_string($plugin->changelog) ? $plugin->changelog : (string) $plugin->changelog;
            }
            if (isset($plugin->faq) && !empty($plugin->faq)) {
                $sections['faq'] = is_string($plugin->faq) ? $plugin->faq : (string) $plugin->faq;
            }
            if (isset($plugin->screenshots) && !empty($plugin->screenshots)) {
                $sections['screenshots'] = is_string($plugin->screenshots) ? $plugin->screenshots : (is_array($plugin->screenshots) ? implode(', ', $plugin->screenshots) : (string) $plugin->screenshots);
            }
            if (isset($plugin->reviews) && !empty($plugin->reviews)) {
                $sections['reviews'] = is_string($plugin->reviews) ? $plugin->reviews : (string) $plugin->reviews;
            }
            if (isset($plugin->other_notes) && !empty($plugin->other_notes)) {
                $sections['other_notes'] = is_string($plugin->other_notes) ? $plugin->other_notes : (string) $plugin->other_notes;
            }
        }
        $info->sections = $sections;

        // Handle banners - ensure all values are strings
        if (isset($plugin->banners)) {
            $info->banners = [];
            if (is_object($plugin->banners)) {
                foreach (get_object_vars($plugin->banners) as $key => $value) {
                    if (!empty($value)) {
                        $info->banners[$key] = is_string($value) ? $value : (string) $value;
                    }
                }
            } elseif (is_array($plugin->banners)) {
                foreach ($plugin->banners as $key => $value) {
                    $info->banners[$key] = is_string($value) ? $value : (string) $value;
                }
            }
        }

        // Handle icons - ensure all values are strings
        if (isset($plugin->icons)) {
            $info->icons = [];
            if (is_object($plugin->icons)) {
                foreach (get_object_vars($plugin->icons) as $key => $value) {
                    if (!empty($value)) {
                        $info->icons[$key] = is_string($value) ? $value : (string) $value;
                    }
                }
            } elseif (is_array($plugin->icons)) {
                foreach ($plugin->icons as $key => $value) {
                    $info->icons[$key] = is_string($value) ? $value : (string) $value;
                }
            }
        }

        // Handle ratings and downloads
        $info->rating = floatval($plugin->rating ?? 0);
        $info->num_ratings = intval($plugin->num_ratings ?? 0);
        $info->downloaded = intval($plugin->downloaded ?? 0);
        $info->last_updated = $plugin->last_updated ?? "";
        $info->active_installs = intval($plugin->active_installs ?? 0);

        // Handle contributors and tags with proper defaults
        $info->contributors = [];
        if (isset($plugin->contributors) && is_array($plugin->contributors) && !empty($plugin->contributors)) {
            $info->contributors = $plugin->contributors;
        } elseif (isset($plugin->contributors) && is_string($plugin->contributors) && !empty($plugin->contributors)) {
            $info->contributors = explode(',', $plugin->contributors);
        } else {
            $info->contributors = $this->getDefaultContributors();
        }
        
        $info->tags = [];
        if (isset($plugin->tags) && is_array($plugin->tags)) {
            $info->tags = $plugin->tags;
        } elseif (isset($plugin->tags) && is_string($plugin->tags)) {
            $info->tags = !empty($plugin->tags) ? explode(',', $plugin->tags) : [];
        }

        // Additional WordPress fields with proper defaults
        $info->donate_link = is_string($plugin->donate_link ?? '') ? $plugin->donate_link : '';
        $info->support_threads = is_numeric($plugin->support_threads ?? 0) ? intval($plugin->support_threads) : 0;
        $info->support_threads_resolved = is_numeric($plugin->support_threads_resolved ?? 0) ? intval($plugin->support_threads_resolved) : 0;
        $info->added = is_string($plugin->added ?? '') ? $plugin->added : '';
        $info->stable_tag = is_string($plugin->stable_tag ?? '') ? $plugin->stable_tag : ($plugin->version ?? '1.0.0');

        // Add custom identifiers
        $info->external_repository = true;
        $info->source = "custom_repository";

        error_log("Ahur System: Successfully prepared plugin info for: " . $info->name);
        
        return $info;
    }

    /**
     * Inject plugins into WordPress plugin search results
     */
    public function injectPlugins($result, $action, $args)
    {
        if ($action !== 'query_plugins') {
            return $result;
        }

        // Only inject for searches that might match our plugins
        if (isset($args->search) && !empty($args->search)) {
            $page = isset($args->page) ? intval($args->page) : 1;
            $per_page = isset($args->per_page) ? intval($args->per_page) : 36;
            
            $search_response = $this->searchPlugins($args->search, $page, $per_page);
            
            if ($search_response && $search_response->success && isset($search_response->data)) {
                foreach ($search_response->data as $plugin) {
                    $formatted_plugin = $this->formatPluginForWordPress($plugin);
                    if ($formatted_plugin) {
                        $result->plugins[] = $formatted_plugin;
                    }
                }
                
                // Update result count
                $result->info['results'] = count($result->plugins);
            }
        }

        return $result;
    }

    /**
     * Format plugin data for WordPress plugin installation interface
     */
    private function formatPluginForWordPress($plugin)
    {
        if (!is_object($plugin)) {
            return null;
        }

        $formatted = new stdClass();
        
        // Basic plugin info
        $formatted->name = $plugin->name ?? 'Unknown Plugin';
        $formatted->slug = $plugin->slug ?? '';
        $formatted->version = $plugin->version ?? '1.0.0';
        $formatted->author = $plugin->author ?? 'Ahur Team';
        $formatted->author_profile = $plugin->author_url ?? '';
        $formatted->homepage = $plugin->homepage ?? $plugin->author_url ?? '';
        $formatted->requires = $plugin->requires ?? '5.0';
        $formatted->tested = $plugin->tested ?? '6.4';
        $formatted->requires_php = $plugin->requires_php ?? '7.4';
        $formatted->download_link = $plugin->downloadlink ?? $plugin->download_url ?? '';
        
        // Check if plugin is already installed
        $is_installed = $this->isPluginInstalled($plugin->slug);
        $formatted->is_installed = $is_installed;
        
        if ($is_installed) {
            // Get installed version
            if (!function_exists('get_plugins')) {
                require_once ABSPATH . 'wp-admin/includes/plugin.php';
            }
            $all_plugins = get_plugins();
            $installed_version = '0.0.0';
            
            foreach ($all_plugins as $plugin_file => $plugin_data) {
                $plugin_slug = dirname($plugin_file);
                $plugin_slug = $plugin_slug === '.' ? basename($plugin_file, '.php') : $plugin_slug;
                
                if (strtolower($plugin_slug) === strtolower($plugin->slug)) {
                    $installed_version = $plugin_data['Version'];
                    break;
                }
            }
            
            $formatted->installed_version = $installed_version;
            $formatted->update_available = version_compare($plugin->version, $installed_version, '>');
        }
        
        // Description and sections - ensure all are strings
        $formatted->short_description = is_string($plugin->short_description ?? '') ? $plugin->short_description : (is_string($plugin->description ?? '') ? $plugin->description : '');
        
        // Handle sections - ensure all are strings
        $formatted->sections = new stdClass();
        if (isset($plugin->sections) && is_object($plugin->sections)) {
            foreach ($plugin->sections as $section_name => $content) {
                if (is_object($content)) {
                    $formatted->sections->$section_name = print_r($content, true);
                } elseif (is_array($content)) {
                    // Handle arrays of objects (like screenshots)
                    $array_strings = [];
                    foreach ($content as $item) {
                        if (is_object($item)) {
                            $array_strings[] = print_r($item, true);
                        } else {
                            $array_strings[] = (string) $item;
                        }
                    }
                    $formatted->sections->$section_name = implode(', ', $array_strings);
                } else {
                    $formatted->sections->$section_name = (string) $content;
                }
            }
        } else {
            // Fallback to direct properties if sections not available
            $formatted->sections->description = is_string($plugin->description ?? '') ? $plugin->description : '';
            $formatted->sections->installation = is_string($plugin->installation ?? '') ? $plugin->installation : '';
            $formatted->sections->changelog = is_string($plugin->changelog ?? '') ? $plugin->changelog : '';
            $formatted->sections->faq = is_string($plugin->faq ?? '') ? $plugin->faq : '';
        }
        
        // Ratings and downloads - ensure proper numeric types
        $formatted->rating = is_numeric($plugin->rating ?? 0) ? floatval($plugin->rating) : 0;
        $formatted->num_ratings = is_numeric($plugin->num_ratings ?? 0) ? intval($plugin->num_ratings) : 0;
        $formatted->downloaded = is_numeric($plugin->downloaded ?? $plugin->download_count ?? 0) ? intval($plugin->downloaded ?? $plugin->download_count) : 0;
        $formatted->active_installs = is_numeric($plugin->active_installs ?? 0) ? intval($plugin->active_installs) : 0;
        
        // Last updated - ensure string
        $formatted->last_updated = is_string($plugin->last_updated ?? '') ? $plugin->last_updated : '';
        
        // Tags - ensure array
        if (isset($plugin->tags) && is_array($plugin->tags)) {
            $formatted->tags = $plugin->tags;
        } elseif (isset($plugin->tags) && is_string($plugin->tags)) {
            $formatted->tags = !empty($plugin->tags) ? explode(',', $plugin->tags) : [];
        } else {
            $formatted->tags = [];
        }
        
        // Banners - ensure all are strings
        $formatted->banners = new stdClass();
        if (isset($plugin->banners) && is_object($plugin->banners)) {
            foreach ($plugin->banners as $banner_name => $url) {
                $formatted->banners->$banner_name = (string) $url;
            }
        } else {
            $formatted->banners = new stdClass();
        }
        
        // Icons (handle both array and object formats)
        if (isset($plugin->icons) && is_object($plugin->icons)) {
            $icons_array = [];
            
            // Map 'default' to standard WordPress icon sizes
            if (isset($plugin->icons->default)) {
                $icons_array['1x'] = $plugin->icons->default;
                $icons_array['2x'] = $plugin->icons->default; // Use same for both sizes
            }
            
            // Handle any other icon keys (like 'svg', 'high', etc.)
            foreach ($plugin->icons as $key => $url) {
                if ($key !== 'default') {
                    $icons_array[$key] = $url;
                }
            }
            
            // Only set icons if we have any, otherwise omit like WordPress.org
            if (!empty($icons_array)) {
                $formatted->icons = $icons_array;
            }
        } elseif (isset($plugin->icons) && is_array($plugin->icons) && !empty($plugin->icons)) {
            $formatted->icons = $plugin->icons;
        }
        // If no icons, don't set the field at all (like WordPress.org)
        
        // Contributors and tags - handle null values
        $formatted->contributors = isset($plugin->contributors) && $plugin->contributors ? (array) $plugin->contributors : $this->getDefaultContributors();
        
        // Support threads
        $formatted->support_threads = isset($plugin->support_threads) && $plugin->support_threads ? (int) $plugin->support_threads : 0;
        $formatted->support_threads_resolved = isset($plugin->support_threads_resolved) && $plugin->support_threads_resolved ? (int) $plugin->support_threads_resolved : 0;
        
        // Add custom identifier
        $formatted->external_repository = true;
        $formatted->source = 'ahur_repository';
        
        return $formatted;
    }

    /**
     * Add custom "اهور" tab to plugin installation page
     */
    public function addAhurTab($tabs)
    {
        $tabs['ahur'] = 'اهور';
        return $tabs;
    }

    /**
     * Get API arguments for Ahur tab
     */
    public function getAhurTabArgs($args)
    {
        return [
            'page' => 1,
            'per_page' => 36,
            'fields' => [
                'last_updated' => true,
                'icons' => true,
                'active_installs' => true,
                'short_description' => true,
                'description' => true,
                'rating' => true,
                'ratings' => true,
                'downloaded' => true,
                'downloadlink' => true,
                'tags' => true,
                'homepage' => true,
                'author' => true,
                'author_profile' => true,
                'versions' => true,
                'requires' => true,
                'tested' => true,
                'requires_php' => true,
                'num_ratings' => true,
                'rating' => true,
            ],
            'locale' => get_locale(),
            'ahur_repository' => true,
        ];
    }

    /**
     * Get default contributors for plugins
     */
    private function getDefaultContributors()
    {
        return [
            'ali_khaleghi' => [
                'profile' => 'mailto:ali@khaleghi.info',
                'avatar' => 'https://www.gravatar.com/avatar/' . md5('ali@khaleghi.info') . '?s=96&d=mm',
                'display_name' => 'Ali Khaleghi'
            ],
            'javad_ahmadi' => [
                'profile' => 'mailto:javad@ahur.ir', 
                'avatar' => 'https://www.gravatar.com/avatar/' . md5('javad@ahur.ir') . '?s=96&d=mm',
                'display_name' => 'Javad Ahmadi'
            ]
        ];
    }

    /**
     * Handle API calls for Ahur tab
     */
    public function handleAhurTabApi($false, $action, $args)
    {
        // Only handle Ahur tab requests
        if (!isset($_GET['tab']) || $_GET['tab'] !== 'ahur') {
            return $false;
        }

        if ($action === 'query_plugins') {
            // Get pagination parameters
            $page = isset($args->page) ? intval($args->page) : 1;
            $per_page = isset($args->per_page) ? intval($args->per_page) : 36;
            
            // Get our repository plugins with pagination
            error_log('Ahur System: Fetching plugins from API: ' . $this->api_base . '/plugins/info');
            $response = $this->fetchPluginsList($page, $per_page);
            
            if (!$response) {
                error_log('Ahur System: No response from API');
                return new WP_Error('no_plugins', 'No plugins found in Ahur repository');
            }
            
            // Check if response has the expected structure
            if (!isset($response->success) || !$response->success || !isset($response->data)) {
                error_log('Ahur System: Invalid API response structure');
                return new WP_Error('invalid_response', 'Invalid API response structure');
            }
            
            $ahur_plugins = $response->data;
            error_log('Ahur System: Found ' . count($ahur_plugins) . ' plugins from repository');

            // Convert our plugins to WordPress format
            $formatted_plugins = [];
            foreach ($ahur_plugins as $plugin) {
                $formatted_plugin = $this->formatPluginForWordPress($plugin);
                if ($formatted_plugin) {
                    $formatted_plugins[] = $formatted_plugin;
                }
            }

            // Create a result object that matches WordPress's expected format
            $result = new stdClass();
            $result->plugins = $formatted_plugins;
            $result->info = [
                'page' => $page,
                'pages' => isset($response->pagination->total_pages) ? $response->pagination->total_pages : 1,
                'results' => count($formatted_plugins),
            ];

            return $result;
        }

        return $false;
    }

    /**
     * Display content for Ahur tab
     */
    public function displayAhurTabContent($page = 1)
    {
        ?>
        <div class="wp-filter">
            <div class="filter-count">
                <span class="count-text"><?php _e('Showing all Ahur repository plugins', 'ahur-system'); ?></span>
            </div>
        </div>
        <?php
        // This will trigger the plugins_api filter and display our plugins
        $api = plugins_api('query_plugins', [
            'page' => $page,
            'per_page' => 36,
            'fields' => [
                'last_updated' => true,
                'icons' => true,
                'active_installs' => true,
                'short_description' => true,
                'description' => true,
                'rating' => true,
                'ratings' => true,
                'downloaded' => true,
                'downloadlink' => true,
                'tags' => true,
                'homepage' => true,
                'author' => true,
                'author_profile' => true,
                'versions' => true,
                'requires' => true,
                'tested' => true,
                'requires_php' => true,
                'num_ratings' => true,
                'rating' => true,
            ],
            'ahur_repository' => true,
        ]);

        if (is_wp_error($api)) {
            ?>
            <div class="error">
                <p><?php _e('Error loading Ahur repository plugins:', 'ahur-system'); ?> <?php echo esc_html($api->get_error_message()); ?></p>
            </div>
            <?php
            return;
        }

        // Display plugins using WordPress's standard display function
        display_plugins_table($api->plugins, $api->info['page'], $api->info['pages']);
    }

    /**
     * Handle installation conflicts
     */
    public function handleInstallation($source, $remote_source, $upgrader, $args = [])
    {
        // Check if this is our repository plugin
        if (!isset($args['hook_extra']) || !isset($args['hook_extra']['plugin'])) {
            return $source;
        }
        
        $plugin_slug = $args['hook_extra']['plugin'];
        error_log("Ahur System: Installing plugin from source: " . $source);
        
        // Remove existing directory if it exists to prevent conflicts
        if (is_dir($source)) {
            $target_dir = WP_PLUGIN_DIR . '/' . basename($source);
            if (is_dir($target_dir)) {
                error_log("Ahur System: Removing existing directory: " . $target_dir);
                $this->removeDirectory($target_dir);
            }
        }
        
        return $source;
    }

    /**
     * Handle post-installation
     */
    public function postInstall($response, $hook_extra, $result)
    {
        // Check if this is our repository plugin installation
        if (isset($hook_extra['plugin']) && isset($result['destination'])) {
            error_log("Ahur System: Plugin installation completed for: " . $hook_extra['plugin']);
            error_log("Ahur System: Installation destination: " . $result['destination']);
            
            // Clear plugin cache to refresh installation status
            wp_cache_flush();
            $this->clearCache();
        }
        
        return $response;
    }

    /**
     * Pre-download validation
     */
    public function preDownload($reply, $package, $upgrader)
    {
        // Add any pre-download validation here
        return $reply;
    }

    /**
     * Force refresh plugin status on Ahur tab
     */
    public function forceRefreshPluginStatus()
    {
        if (isset($_GET['tab']) && $_GET['tab'] === 'ahur' && isset($_GET['page']) && $_GET['page'] === 'plugin-install') {
            // Clear WordPress plugin cache
            wp_cache_flush();
            delete_transient('update_plugins');
            error_log("Ahur System: Forced refresh of plugin status on Ahur tab");
        }
    }

    /**
     * Clear plugin info cache when needed
     */
    public function clearPluginInfoCache()
    {
        // Check if we're on plugin info page
        if (isset($_GET['plugin']) && !empty($_GET['plugin'])) {
            $plugin_slug = sanitize_text_field($_GET['plugin']);
            
            // Clear WordPress plugin info cache
            wp_cache_delete('plugin_' . $plugin_slug . '_info', 'plugins');
            
            // Clear our own cache for this plugin
            $this->clearPluginSpecificCache($plugin_slug);
        }
    }
    
    /**
     * Clear cache for specific plugin
     */
    private function clearPluginSpecificCache($slug)
    {
        $cache_keys = [
            "ahur_" . md5($this->api_base . '/plugins?query=' . $slug),
            "ahur_" . md5($this->api_base . '/plugins?query=' . strtolower($slug)),
        ];
        
        foreach ($cache_keys as $cache_key) {
            delete_transient($cache_key);
        }
    }

    /**
     * Clear all caches
     */
    public function clearCache()
    {
        // Clear plugins list cache
        delete_transient("ahur_plugins_list_" . md5($this->api_base . '/plugins/info'));
        
        // Clear individual plugin caches
        global $wpdb;
        $cache_keys = $wpdb->get_col(
            "SELECT option_name FROM {$wpdb->options} WHERE option_name LIKE '_transient_ahur_%'"
        );
        
        foreach ($cache_keys as $cache_key) {
            delete_transient(str_replace('_transient_', '', $cache_key));
        }
        
        // Clear WordPress plugin caches
        wp_cache_flush();
    }

    /**
     * Force plugin info check for our repository plugins
     */
    public function forcePluginInfoCheck($transient)
    {
        // This ensures our plugins are always checked for info
        return $transient;
    }

    /**
     * Modify plugin install status after API response
     */
    public function modifyPluginInstallStatus($result, $action, $args)
    {
        // Only modify our Ahur tab results
        if ($action !== 'query_plugins' || !isset($_GET['tab']) || $_GET['tab'] !== 'ahur') {
            return $result;
        }
        
        if (!isset($result->plugins) || !is_array($result->plugins)) {
            return $result;
        }
        
        // Get all installed plugins
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        $all_plugins = get_plugins();

        // Debug: Log all installed plugins for troubleshooting
        error_log("Ahur System: Checking installation status for plugins:");
        foreach ($all_plugins as $plugin_file => $plugin_data) {
            $plugin_slug = dirname($plugin_file);
            $plugin_slug = $plugin_slug === '.' ? basename($plugin_file, '.php') : $plugin_slug;
            error_log("Ahur System: Installed plugin: {$plugin_file} -> slug: {$plugin_slug}");
        }
        
        foreach ($result->plugins as &$plugin) {
            // Only modify our repository plugins
            if (!isset($plugin->source) || $plugin->source !== 'ahur_repository') {
                continue;
            }
            
            // Check if plugin is installed
            $is_installed = false;
            $installed_version = '0.0.0';
            
            foreach ($all_plugins as $plugin_file => $plugin_data) {
                $plugin_slug = dirname($plugin_file);
                $plugin_slug = $plugin_slug === '.' ? basename($plugin_file, '.php') : $plugin_slug;
                $plugin_slug_lower = strtolower($plugin_slug);
                $repo_slug_lower = strtolower($plugin->slug);

                // Try multiple matching strategies (same as isPluginInstalled method)
                if ($plugin_slug_lower === $repo_slug_lower ||
                    str_replace(['-', '_'], '', $plugin_slug_lower) === str_replace(['-', '_'], '', $repo_slug_lower) ||
                    strpos($plugin_slug_lower, $repo_slug_lower) !== false ||
                    strpos($repo_slug_lower, $plugin_slug_lower) !== false) {
                    $is_installed = true;
                    $installed_version = $plugin_data['Version'];
                    error_log("Ahur System: Plugin {$plugin->slug} is installed as {$plugin_file} (matched via comprehensive logic)");
                    break;
                }
            }
            
            if ($is_installed) {
                // Set WordPress properties for installed plugins
                $plugin->installed = true;
                $plugin->version_installed = $installed_version;
                $plugin->update_available = version_compare($plugin->version, $installed_version, '>');
                
                // Set action based on update availability
                if ($plugin->update_available) {
                    $plugin->action = 'update';
                } else {
                    $plugin->action = 'active';
                }
            } else {
                $plugin->installed = false;
                $plugin->action = 'install';
            }
        }
        
        return $result;
    }

    /**
     * Fix plugin action links for installed plugins
     */
    public function fixPluginActionLinks($links, $plugin)
    {
        // Only modify our repository plugins
        if (!isset($plugin->source) || $plugin->source !== 'ahur_repository') {
            return $links;
        }
        
        // Check if plugin is installed
        $is_installed = $this->isPluginInstalled($plugin->slug);
        
        if ($is_installed) {
            // Get installed version
            if (!function_exists('get_plugins')) {
                require_once ABSPATH . 'wp-admin/includes/plugin.php';
            }
            $all_plugins = get_plugins();
            $installed_version = '0.0.0';
            
            foreach ($all_plugins as $plugin_file => $plugin_data) {
                $plugin_slug = dirname($plugin_file);
                $plugin_slug = $plugin_slug === '.' ? basename($plugin_file, '.php') : $plugin_slug;
                
                if (strtolower($plugin_slug) === strtolower($plugin->slug)) {
                    $installed_version = $plugin_data['Version'];
                    break;
                }
            }
            
            // Check if update is available
            $update_available = version_compare($plugin->version, $installed_version, '>');
            
            if ($update_available) {
                // Show update button
                $links = [
                    'update' => sprintf(
                        '<a class="button button-primary update-now" href="%s" data-slug="%s" data-name="%s">%s</a>',
                        admin_url('update-core.php'),
                        esc_attr($plugin->slug),
                        esc_attr($plugin->name),
                        __('Update Now', 'ahur-system')
                    )
                ];
            } else {
                // Show active button (disabled)
                $links = [
                    'active' => sprintf(
                        '<button class="button button-disabled" disabled>%s</button>',
                        __('Active', 'ahur-system')
                    )
                ];
            }
        }
        
        return $links;
    }

    /**
     * Remove directory recursively
     */
    private function removeDirectory($dir)
    {
        if (!is_dir($dir)) {
            return true;
        }

        $files = array_diff(scandir($dir), ['.', '..']);

        foreach ($files as $file) {
            $path = $dir . '/' . $file;
            if (is_dir($path)) {
                $this->removeDirectory($path);
            } else {
                unlink($path);
            }
        }

        return rmdir($dir);
    }
}