<?php
/**
 * Theme Repository Manager
 *
 * Handles all theme-related operations for the Ahur repository
 *
 * @package AhurSystem
 * @author Ahur Team
 * @version 0.0.9
 */

// Prevent direct access
if (!defined("ABSPATH")) {
    exit();
}

class AhurThemeRepository
{
    /**
     * API base URL
     */
    private $api_base;

    /**
     * Cache time in hours
     */
    private $cache_time;

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->api_base = get_option(
            "ahur_system_api_url",
            AHUR_SYSTEM_API_BASE,
        );
        $this->cache_time = (int) get_option("ahur_system_cache_time", 6);

        error_log('Ahur System: ThemeRepository constructor called');

        // Add filters early
        add_filter("themes_api", [$this, "handleAhurTabApi"], 9, 3);
        add_filter("themes_api", [$this, "getThemeInfo"], 10, 3);
        add_filter("themes_api_result", [$this, "injectThemes"], 999, 3);
    }

    /**
     * Initialize theme hooks
     */
    public function initHooks()
    {
        error_log('Ahur System: ThemeRepository initHooks called');

        // Theme updates and info
        add_filter("pre_set_site_transient_update_themes", [$this, "checkThemeUpdates"]);
        // Filters already added in constructor
        error_log('Ahur System: Theme filters added');

        // Installation handling
        add_filter("upgrader_source_selection", [$this, "handleThemeInstallation"], 10, 4);

        // Cache clearing
        add_action("upgrader_process_complete", [$this, "clearCache"], 10, 2);
        add_action("switch_theme", [$this, "clearCache"]);

        // Custom Ahur tab for theme installation - use correct WordPress hooks
        add_filter("install_themes_tabs", [$this, "addAhurTab"], 1);
        add_filter("install_themes_table_api_args_ahur", [$this, "getAhurTabArgs"], 1);
        add_filter("themes_api", [$this, "handleAhurTabApi"], 9, 3); // Higher priority to override default

        // Override the display function for Ahur tab
        add_action("install_themes_ahur", [$this, "displayAhurThemesSimple"], 999);

        // Handle AJAX requests for theme browsing
        add_action('wp_ajax_query-themes', [$this, 'handleAjaxThemeQuery'], 1);

        // Also inject themes into search results
        add_filter('themes_api_result', [$this, 'injectThemesIntoSearch'], 999, 3);



        // Also try to inject into theme search results directly
        add_filter("themes_api_result", [$this, "injectThemes"], 999, 3);

        // Handle direct theme URLs by redirecting to the tab
        add_action('init', [$this, 'handleDirectThemeUrl']);

        // Intercept WordPress.org API calls for our themes
        add_filter('http_request_args', [$this, 'interceptThemeApiRequest'], 10, 2);
        add_filter('http_response', [$this, 'interceptThemeApiResponse'], 10, 3);

        // Debug: Log that hooks are being registered
        error_log('Ahur System: Theme hooks registered');

        // Test hook to see if we're on theme installer page
        add_action('admin_init', function() {
            $current_screen = get_current_screen();
            if ($current_screen && $current_screen->base === 'theme-install') {
                error_log('Ahur System: On theme installer screen');
            }
            if (isset($_GET['tab']) && $_GET['tab'] === 'ahur') {
                error_log('Ahur System: Ahur tab requested');
            }
            // Log all GET params for debugging
            error_log('Ahur System: Current GET params: ' . print_r($_GET, true));
        });
    }

    /**
     * Fetch themes list from API
     */
    public function fetchThemesList()
    {
        $url = $this->api_base . '/themes/info';
        $cache_key = "ahur_themes_list_" . md5($url);

        // Check cache first
        $cached = get_transient($cache_key);
        if ($cached !== false) {
            return $cached;
        }

        $response = wp_remote_get($url, [
            "timeout" => 10,
            "headers" => [
                "User-Agent" => "WordPress/" . get_bloginfo("version") . "; " . get_bloginfo("url"),
            ],
        ]);

        if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
            error_log('Ahur System: Themes API request failed: ' . (is_wp_error($response) ? $response->get_error_message() : 'HTTP ' . wp_remote_retrieve_response_code($response)));
            return false;
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body);

        if (!$data) {
            error_log('Ahur System: Failed to decode themes JSON response');
            return false;
        }

        // Cache the full response for 1 hour
        set_transient($cache_key, $data, HOUR_IN_SECONDS);

        return $data;
    }

    /**
     * Unified method to fetch theme data from API with caching
     */
    private function fetchFromAPI($endpoint, $search = null, $useCache = true)
    {
        $url = $this->api_base . $endpoint;
        if ($search) {
            $url .= "?search=" . urlencode($search);
        }

        $cache_key = "ahur_theme_" . md5($url);

        if ($useCache) {
            $cached = get_transient($cache_key);
            if ($cached !== false) {
                return $cached;
            }
        }

        $response = wp_remote_get($url, [
            "timeout" => 10,
            "headers" => [
                "User-Agent" => "WordPress/" . get_bloginfo("version") . "; " . get_bloginfo("url"),
            ],
        ]);

        if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
            return false;
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body);

        if (!$data || !isset($data->data)) {
            return false;
        }

        $result = is_array($data->data) ? $data->data : [$data->data];

        if ($useCache) {
            set_transient($cache_key, $result, $this->cache_time * HOUR_IN_SECONDS);
        }

        return $result;
    }

    /**
     * Smart sanitization - convert objects to proper WordPress format
     */
    private function sanitizeData($data)
    {
        if (!is_object($data)) {
            return $data;
        }

        $sanitized = new stdClass();

        foreach (get_object_vars($data) as $key => $value) {
            if (is_array($value)) {
                // Convert arrays to strings or skip
                if ($key === 'tags' || $key === 'versions') {
                    $sanitized->$key = implode(', ', $value);
                } elseif ($key === 'icons' || $key === 'banners') {
                    $sanitized->$key = $this->sanitizeBanners($value);
                } else {
                    $sanitized->$key = ''; // Skip other arrays
                }
            } elseif (is_object($value)) {
                // Handle nested objects (like sections)
                if ($key === 'sections') {
                    $sanitized->$key = $this->sanitizeSections($value);
                } else {
                    $sanitized->$key = ''; // Skip other objects
                }
            } else {
                $sanitized->$key = $value;
            }
        }

        return $sanitized;
    }

    /**
     * Sanitize sections object to array
     */
    private function sanitizeSections($sections)
    {
        if (!is_object($sections)) {
            return $sections;
        }

        $sanitized = [];
        foreach (get_object_vars($sections) as $key => $value) {
            if (!is_array($value) && !is_object($value)) {
                $sanitized[$key] = $value;
            } else {
                $sanitized[$key] = ''; // Convert complex objects to empty strings
            }
        }

        return $sanitized;
    }

    /**
     * Sanitize banners/icons array
     */
    private function sanitizeBanners($banners)
    {
        if (!is_array($banners)) {
            return $banners;
        }

        $sanitized = [];
        foreach ($banners as $key => $value) {
            if (!is_array($value) && !is_object($value)) {
                $sanitized[$key] = $value;
            }
        }

        return $sanitized;
    }

    /**
     * Check if theme is installed
     */
    private function isThemeInstalled($slug)
    {
        $all_themes = wp_get_themes();
        return isset($all_themes[$slug]);
    }

    /**
     * Check for theme updates
     */
    public function checkThemeUpdates($transient)
    {
        if (empty($transient->checked)) {
            return $transient;
        }

        $response = $this->fetchThemesList();

        if (!$response || !$response->success || !isset($response->data)) {
            return $transient;
        }

        $themes = $response->data;

        foreach ($themes as $theme) {
            $theme_slug = $theme->slug;

            // Check if theme is installed
            if (isset($transient->checked[$theme_slug])) {
                $installed_version = $transient->checked[$theme_slug];
                $remote_version = $theme->version ?? "unknown";

                if (version_compare($remote_version, $installed_version, ">")) {
                    $update = [
                        'theme' => $theme_slug,
                        'new_version' => $remote_version,
                        'url' => $theme->author_url ?? '',
                        'package' => $theme->downloadlink ?? '',
                        'requires' => $theme->requires ?? '',
                        'requires_php' => $theme->requires_php ?? '',
                        'tested' => $theme->tested ?? '',
                    ];

                    $transient->response[$theme_slug] = $update;
                }
            }
        }

        return $transient;
    }

    /**
     * Get theme information
     */
    public function getThemeInfo($false, $action, $response)
    {
        error_log('Ahur System: getThemeInfo called with action: ' . $action);

        if ($action !== "theme_information") {
            error_log('Ahur System: Action is not theme_information, returning false');
            return $false;
        }

        $theme_slug = is_array($response) ? $response['slug'] : $response->slug;
        error_log('Ahur System: getThemeInfo called for slug: ' . $theme_slug);

        // Fetch the specific theme
        $url = $this->api_base . '/themes/' . $theme_slug;
        $cache_key = "ahur_theme_" . md5($url);

        $cached = get_transient($cache_key);
        if ($cached !== false) {
            $theme = $cached;
        } else {
            $response = wp_remote_get($url, [
                "timeout" => 10,
                "headers" => [
                    "User-Agent" => "WordPress/" . get_bloginfo("version") . "; " . get_bloginfo("url"),
                ],
            ]);

            if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
                error_log('Ahur System: Theme API request failed: ' . (is_wp_error($response) ? $response->get_error_message() : 'HTTP ' . wp_remote_retrieve_response_code($response)));
                return $false;
            }

            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body);

            if (!$data || !$data->success || !isset($data->data)) {
                error_log('Ahur System: Failed to decode theme JSON response');
                return $false;
            }

            $theme_data = $data->data;
            if (is_array($theme_data)) {
                if (isset($theme_data[0]) && is_array($theme_data[0])) {
                    $theme = (object) $theme_data[0];
                } elseif (isset($theme_data[0]) && is_object($theme_data[0])) {
                    $theme = $theme_data[0];
                } else {
                    // Assume $theme_data is the theme data array directly
                    $theme = (object) $theme_data;
                }
            } elseif (is_object($theme_data)) {
                $theme = $theme_data;
            } else {
                error_log('Ahur System: Invalid theme data type');
                return $false;
            }
            set_transient($cache_key, $theme, $this->cache_time * HOUR_IN_SECONDS);
        }

        if (!$theme) {
            error_log('Ahur System: Theme not found in repository');
            return $false;
        }

        error_log('Ahur System: Found theme: ' . $theme->name);

        // Create a new object for the response
        $result = new stdClass();

        // Sanitize and format theme data
        $theme = $this->sanitizeData($theme);

        // Map theme data to WordPress format
        $result->name = $theme->name ?? "Unknown Theme";
        $result->slug = $theme->slug ?? "";
        $result->version = $theme->version ?? "1.0.0";
        $result->author = $theme->author ?? "";
        $result->homepage = $theme->author_url ?? "";
        $result->requires = $theme->requires ?? "";
        $result->tested = $theme->tested ?? "";
        $result->requires_php = $theme->requires_php ?? "";
        $result->download_link = $theme->downloadlink ?? "";

        // Handle sections
        $result->sections = (array) ($theme->sections ?? []);

        // Handle ratings and downloads
        $result->rating = floatval($theme->rating ?? 0);
        $result->num_ratings = intval($theme->num_ratings ?? 0);
        $result->downloaded = intval($theme->downloaded ?? 0);
        $result->last_updated = $theme->last_updated ?? "";

        // Handle tags
        $result->tags = $theme->tags ?? [];

        // Handle screenshots
        if (isset($theme->screenshot_url) && $theme->screenshot_url) {
            // Convert relative URL to full HTTPS URL
            if (strpos($theme->screenshot_url, 'http') !== 0) {
                // Remove '/api' from base URL to get the main site URL
                $site_url = str_replace('/api', '', $this->api_base);
                $result->screenshot_url = $site_url . '/' . $theme->screenshot_url;
            } else {
                $result->screenshot_url = $theme->screenshot_url;
            }
            // Ensure HTTPS
            $result->screenshot_url = str_replace('http://', 'https://', $result->screenshot_url);
        } else {
            $result->screenshot_url = '';
        }

        // Handle preview URL
        if (isset($theme->preview_url) && $theme->preview_url) {
            $result->preview_url = $theme->preview_url;
        } else {
            $result->preview_url = '';
        }

        // Handle icons and banners as arrays
        $result->icons = (array) ($theme->icons ?? []);
        $result->banners = (array) ($theme->banners ?? []);

        // Add custom identifiers
        $result->external_repository = true;
        $result->source = "custom_repository";

        return $result;
    }

    /**
     * Inject themes into WordPress theme search results
     */
    public function injectThemes($result, $action, $args)
    {
        error_log('Ahur System: injectThemes called with action: ' . $action);

        // Handle theme_information for direct theme URLs
        if ($action === 'theme_information') {
            error_log('Ahur System: Processing theme_information, result type: ' . gettype($result));
            if (is_wp_error($result)) {
                error_log('Ahur System: Result is WP_Error: ' . $result->get_error_message());
            }
            // Check if this is a request for our theme
            $slug = is_object($args) ? ($args->slug ?? '') : ($args['slug'] ?? '');

            // Fetch all themes and find the matching one
            $response_data = $this->fetchThemesList();
            if ($response_data && $response_data->success && isset($response_data->data)) {
                foreach ($response_data->data as $theme) {
                    $theme = is_array($theme) ? (object) $theme : $theme;
                    if ($theme->slug === $slug) {
                        error_log('Ahur System: Found theme for theme_information: ' . $theme->name);

                        // Create the theme object
                        $result = new stdClass();

                        // Sanitize and format theme data
                        $theme = $this->sanitizeData($theme);

                        // Map theme data to WordPress format
                        $result->name = $theme->name ?? "Unknown Theme";
                        $result->slug = $theme->slug ?? "";
                        $result->version = $theme->version ?? "1.0.0";
                        $result->author = $theme->author ?? "";
                        $result->homepage = $theme->author_url ?? "";
                        $result->requires = $theme->requires ?? "";
                        $result->tested = $theme->tested ?? "";
                        $result->requires_php = $theme->requires_php ?? "";
                        $result->download_link = $theme->downloadlink ?? "";

                        // Handle sections
                        $result->sections = (array) ($theme->sections ?? []);

                        // Handle ratings and downloads
                        $result->rating = floatval($theme->rating ?? 0);
                        $result->num_ratings = intval($theme->num_ratings ?? 0);
                        $result->downloaded = intval($theme->downloaded ?? 0);
                        $result->last_updated = $theme->last_updated ?? "";

                        // Handle tags
                        $result->tags = $theme->tags ?? [];

                        // Handle screenshots
                        if (isset($theme->screenshot_url) && $theme->screenshot_url) {
                            // Convert relative URL to full HTTPS URL
                            if (strpos($theme->screenshot_url, 'http') !== 0) {
                                // Remove '/api' from base URL to get the main site URL
                                $site_url = str_replace('/api', '', $this->api_base);
                                $result->screenshot_url = $site_url . '/' . $theme->screenshot_url;
                            } else {
                                $result->screenshot_url = $theme->screenshot_url;
                            }
                            // Ensure HTTPS
                            $result->screenshot_url = str_replace('http://', 'https://', $result->screenshot_url);
                        } else {
                            $result->screenshot_url = '';
                        }

                        // Handle preview URL
                        if (isset($theme->preview_url) && $theme->preview_url) {
                            $result->preview_url = $theme->preview_url;
                        } else {
                            $result->preview_url = '';
                        }

                        // Add custom identifiers
                        $result->external_repository = true;
                        $result->source = "custom_repository";

                        return $result;
                    }
                }
            }

            // If not our theme, return the original result
            return $result;
        }

        if ($action !== 'query_themes' || !is_object($result) || is_wp_error($result) || !isset($result->themes) || !is_array($result->themes)) {
            return $result;
        }

        // Only inject for searches that might match our themes
        if (isset($args->search) && !empty($args->search)) {
            $search_lower = strtolower($args->search);
            error_log('Ahur System: Checking search: ' . $search_lower);
            if (strpos($search_lower, 'kerman') !== false || strpos($search_lower, 'اهور') !== false || strpos($search_lower, 'kermanshop') !== false) {
                error_log('Ahur System: Fetching themes for search: ' . $args->search);
                // Fetch themes matching the search
                $themes = $this->fetchFromAPI('/themes/info', $args->search);
                error_log('Ahur System: API response: ' . (is_object($themes) ? 'object' : gettype($themes)));
                if ($themes && is_array($themes) && !empty($themes)) {
                    error_log('Ahur System: Injecting ' . count($themes) . ' themes');
                    foreach ($themes as $theme) {
                        $formatted_theme = $this->formatThemeForWordPress($theme);
                        if ($formatted_theme) {
                            $result->themes[] = $formatted_theme;
                        }
                    }

                    // Update result count
                    $result->info['results'] = count($result->themes);
                } else {
                    error_log('Ahur System: No themes to inject');
                }
            } else {
                error_log('Ahur System: Search does not match condition');
            }
        } else {
            error_log('Ahur System: No search in args');
        }

        return $result;
    }

    /**
     * Capture and replace WordPress theme output
     */
    public function captureThemeOutput($content)
    {
        // Check if this is the "no themes found" message
        if (strpos($content, 'هیچ پوسته‌ای یافت نشد') !== false || strpos($content, 'No themes found') !== false) {
            error_log('Ahur System: Replacing "no themes found" with our content');

            // Get our themes and display them instead
            $response = $this->fetchThemesList();

            if ($response && $response->success && isset($response->data)) {
                ob_clean();
                $this->displayAhurThemesSimple();
                return '';
            }
        }

        return $content;
    }

    /**
     * Inject themes into search results (for AJAX requests)
     */
    public function injectThemesIntoSearch($result, $action, $args)
    {
        if ($action !== 'query_themes') {
            return $result;
        }

        // Check if this is a search that should include our themes
        $search = isset($args->search) ? $args->search : '';
        $tab = isset($args->tab) ? $args->tab : '';

        if ($tab === 'ahur' || ($search && (strpos(strtolower($search), 'kermanshop') !== false || strpos(strtolower($search), 'اهور') !== false))) {
            error_log('Ahur System: Injecting themes into search results for: ' . $search);

            // Get our themes
            $response = $this->fetchThemesList();

            if ($response && $response->success && isset($response->data)) {
                foreach ($response->data as $theme) {
                    $formatted_theme = $this->formatThemeForWordPress($theme);
                    if ($formatted_theme) {
                        $result->themes[] = $formatted_theme;
                    }
                }

                // Update result count
                $result->info['results'] = count($result->themes);
            }
        }

        return $result;
    }

    /**
     * Add custom "اهور" tab to theme installation page
     */
    public function addAhurTab($tabs)
    {
        error_log('Ahur System: Adding theme tab to: ' . print_r($tabs, true));
        $tabs['ahur'] = 'اهور';
        error_log('Ahur System: Theme tabs after adding: ' . print_r($tabs, true));
        return $tabs;
    }

    /**
     * Get API arguments for Ahur tab
     */
    public function getAhurTabArgs($args)
    {
        return [
            'page' => 1,
            'per_page' => 36,
            'fields' => [
                'last_updated' => true,
                'icons' => true,
                'active_installs' => true,
                'short_description' => true,
                'description' => true,
                'rating' => true,
                'ratings' => true,
                'downloaded' => true,
                'downloadlink' => true,
                'tags' => true,
                'homepage' => true,
                'author' => true,
                'author_profile' => true,
                'versions' => true,
                'requires' => true,
                'tested' => true,
                'requires_php' => true,
                'num_ratings' => true,
                'rating' => true,
                'screenshot_url' => true,
                'preview_url' => true,
            ],
            'locale' => get_locale(),
            'ahur_repository' => true,
        ];
    }

    /**
     * Handle API calls for Ahur tab
     */
    public function handleAhurTabApi($false, $action, $args)
    {
        error_log('Ahur System: handleAhurTabApi called with action: ' . $action . ', tab: ' . ($_GET['tab'] ?? 'none'));

        // Only handle Ahur tab requests
        if (!isset($_GET['tab']) || $_GET['tab'] !== 'ahur') {
            error_log('Ahur System: Not Ahur tab, returning false');
            return $false;
        }

        error_log('Ahur System: Handling Ahur tab for action: ' . $action);

        if ($action === 'query_themes') {
            // Get our repository themes
            error_log('Ahur System: Fetching themes from API: ' . $this->api_base . '/themes/info');
            $response = $this->fetchThemesList();

            if (!$response) {
                error_log('Ahur System: No themes response from API');
                return new WP_Error('no_themes', 'No themes found in Ahur repository');
            }

            // Check if response has the expected structure
            if (!is_object($response) || !property_exists($response, 'success') || !$response->success || !isset($response->data)) {
                error_log('Ahur System: Invalid themes API response structure');
                return new WP_Error('invalid_response', 'Invalid API response structure');
            }

            $ahur_themes = $response->data;
            error_log('Ahur System: Found ' . count($ahur_themes) . ' themes from repository');

            // Convert our themes to WordPress format
            $formatted_themes = [];
            foreach ($ahur_themes as $theme) {
                $formatted_theme = $this->formatThemeForWordPress($theme);
                if ($formatted_theme) {
                    $formatted_themes[] = $formatted_theme;
                }
            }

            // Create a result object that matches WordPress's expected format
            $result = new stdClass();
            $result->themes = $formatted_themes;
            $result->info = [
                'page' => 1,
                'pages' => 1,
                'results' => count($formatted_themes),
            ];

            return $result;
        }

        return $false;
    }



    /**
     * Handle AJAX theme queries
     */
    public function handleAjaxThemeQuery()
    {
        error_log('Ahur System: Handling AJAX theme query: ' . print_r($_REQUEST, true));

        // Check if this is our Ahur tab request (could be POST or GET)
        $tab = isset($_REQUEST['tab']) ? $_REQUEST['tab'] : '';
        $browse = isset($_REQUEST['browse']) ? $_REQUEST['browse'] : '';
        $search = isset($_REQUEST['s']) ? $_REQUEST['s'] : '';
        $theme = isset($_REQUEST['theme']) ? $_REQUEST['theme'] : '';

        // Also check inside the 'request' array (WordPress AJAX format)
        if (isset($_REQUEST['request'])) {
            $request = $_REQUEST['request'];
            if (is_array($request)) {
                $tab = isset($request['tab']) ? $request['tab'] : $tab;
                $browse = isset($request['browse']) ? $request['browse'] : $browse;
                $search = isset($request['search']) ? $request['search'] : $search;
                $theme = isset($request['theme']) ? $request['theme'] : $theme;
            }
        }

        // If theme is specified and it's our theme, treat as Ahur tab
        if ($theme) {
            $response = $this->fetchThemesList();
            if ($response && $response->success && isset($response->data)) {
                foreach ($response->data as $t) {
                    if ($t->slug === $theme) {
                        $tab = 'ahur';
                        break;
                    }
                }
            }
        }

        // Check for Ahur tab in both 'tab' and 'browse' parameters
        if ($tab === 'ahur' || $browse === 'ahur' || $browse === 'tab=ahur' || ($search && (strpos(strtolower($search), 'kermanshop') !== false || strpos(strtolower($search), 'اهور') !== false))) {
            error_log('Ahur System: Processing Ahur tab AJAX request');

            // Get our themes
            $response = $this->fetchThemesList();

            if (!$response || !$response->success || !isset($response->data)) {
                wp_send_json_error(['message' => __('Error loading Ahur repository themes.', 'ahur-system')]);
                return;
            }

            $ahur_themes = $response->data;

            // Format themes for WordPress
            $formatted_themes = [];
            foreach ($ahur_themes as $theme) {
                $formatted_theme = $this->formatThemeForWordPress($theme);
                if ($formatted_theme) {
                    $formatted_themes[] = $formatted_theme;
                }
            }

            // Return in WordPress AJAX format
            wp_send_json_success([
                'themes' => $formatted_themes,
                'info' => [
                    'page' => 1,
                    'pages' => 1,
                    'results' => count($formatted_themes),
                ]
            ]);
        }
    }

    /**
     * Display Ahur themes - simple approach without WordPress conflicts
     */
    public function displayAhurThemesSimple($page = 1)
    {
        error_log('Ahur System: Displaying Ahur themes with simple approach');

        // Clear cache to ensure fresh data
        $this->clearCache();

        // Get our themes directly
        $response = $this->fetchThemesList();

        if (!$response || !$response->success || !isset($response->data)) {
            echo '<div class="error"><p>' . __('Error loading Ahur repository themes.', 'ahur-system') . '</p></div>';
            return;
        }

        $ahur_themes = $response->data;

        // Check if requesting a specific theme
        $specific_theme = null;
        if (isset($_GET['theme'])) {
            $requested_slug = sanitize_text_field($_GET['theme']);
            foreach ($ahur_themes as $theme) {
                if ($theme->slug === $requested_slug) {
                    $specific_theme = $theme;
                    break;
                }
            }
        }

        // Simple output without WordPress theme display functions
        echo '<div class="theme-browser rendered">';
        echo '<div class="themes wp-clearfix">';

        if ($specific_theme) {
            // Display only the specific theme
            $formatted_theme = $this->formatThemeForWordPress($specific_theme);
            if ($formatted_theme) {
                $this->displayThemeCard($formatted_theme);
            }
        } else {
            // Display all themes
            foreach ($ahur_themes as $theme) {
                $formatted_theme = $this->formatThemeForWordPress($theme);
                if ($formatted_theme) {
                    $this->displayThemeCard($formatted_theme);
                }
            }
        }

        echo '</div>';
        echo '</div>';

        // Add custom styling
        echo '<style>
        .theme-browser.rendered .theme {
            width: 300px;
            margin: 0 10px 10px 0;
            padding: 15px;
            background: #fff;
            border: 1px solid #ddd;
            border-radius: 3px;
            position: relative;
            cursor: pointer;
            float: left;
        }

        .theme-browser.rendered .theme:hover {
            border-color: #0073aa;
            box-shadow: 0 1px 1px rgba(0,0,0,.04);
        }

        .theme-screenshot {
            height: 180px;
            overflow: hidden;
            margin: -15px -15px 15px -15px;
            position: relative;
        }

        .theme-screenshot img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }

        .theme-name {
            margin: 0 0 10px 0;
            font-size: 16px;
            font-weight: 600;
        }

        .theme-version {
            font-size: 12px;
            color: #666;
            font-weight: normal;
        }

        .theme-author {
            font-size: 13px;
            color: #666;
            margin-bottom: 10px;
        }

        .theme-description {
            font-size: 13px;
            line-height: 1.4;
            margin-bottom: 10px;
            color: #555;
        }

        .theme-tags {
            font-size: 12px;
            color: #888;
        }

        .theme-actions {
            position: absolute;
            top: 15px;
            right: 15px;
            background: rgba(255,255,255,0.9);
            padding: 5px;
            border-radius: 3px;
            opacity: 0;
            transition: opacity 0.2s;
        }

        .theme:hover .theme-actions {
            opacity: 1;
        }

        .theme-actions .button {
            display: block;
            margin-bottom: 5px;
            white-space: nowrap;
        }

        .theme-actions .button:last-child {
            margin-bottom: 0;
        }

        .theme-actions .button.disabled {
            cursor: not-allowed;
            opacity: 0.6;
        }

        .wp-clearfix::after {
            content: "";
            display: table;
            clear: both;
        }
        </style>';

        // Add JavaScript to prevent AJAX override
        echo '<script>
        jQuery(document).ready(function($) {
            // Check if we are on the Ahur tab
            if (window.location.search.indexOf("tab=ahur") !== -1) {
                console.log("Ahur tab detected - preventing AJAX override");

                // Prevent WordPress theme browser from making AJAX calls
                $(document).ajaxSend(function(event, xhr, settings) {
                    if (settings.url && settings.url.indexOf("admin-ajax.php") !== -1 && settings.data && settings.data.indexOf("query-themes") !== -1) {
                        console.log("Blocking theme query AJAX call for Ahur tab");
                        event.preventDefault();
                        return false;
                    }
                });

                // Prevent theme browser initialization and queries
                if (typeof wp !== "undefined" && wp.themes) {
                    // Override the query function to prevent AJAX calls
                    var originalQuery = wp.themes.query;
                    wp.themes.query = function() {
                        console.log("Ahur System: Preventing theme query AJAX call");
                        return jQuery.Deferred().resolve({themes: [], info: {results: 0}}).promise();
                    };

                    // Clear existing theme data
                    if (wp.themes.data) {
                        wp.themes.data.themes = [];
                    }
                }
            }
        });
        </script>';


    }

    /**
     * Display a single theme card
     */
    private function displayThemeCard($theme)
    {
        ?>
        <div class="theme" tabindex="0">
            <div class="theme-screenshot">
                <?php if (!empty($theme->screenshot_url)): ?>
                    <img src="<?php echo esc_url($theme->screenshot_url); ?>" alt="<?php echo esc_attr($theme->name); ?>" />
                <?php else: ?>
                    <img src="<?php echo esc_url(admin_url('images/theme-featured.png')); ?>" alt="<?php echo esc_attr($theme->name); ?>" />
                <?php endif; ?>
            </div>

            <div class="theme-id-container">
                <h2 class="theme-name">
                    <?php echo esc_html($theme->name); ?>
                    <?php if ($this->isThemeInstalled($theme->slug)): ?>
                        <span class="theme-version"><?php printf(__('Version: %s', 'ahur-system'), esc_html($theme->version)); ?></span>
                    <?php endif; ?>
                </h2>

                <div class="theme-actions">
                    <?php if ($this->isThemeInstalled($theme->slug)): ?>
                        <a class="button button-primary disabled" href="#"><?php _e('Installed', 'ahur-system'); ?></a>
                    <?php else: ?>
                        <a class="button button-primary" href="<?php echo esc_url(wp_nonce_url(admin_url('update.php?action=install-theme&theme=' . $theme->slug), 'install-theme_' . $theme->slug)); ?>">
                            <?php _e('Install Now', 'ahur-system'); ?>
                        </a>
                    <?php endif; ?>
                    <a class="button" href="<?php echo esc_url($theme->preview_url); ?>" target="_blank">
                        <?php _e('Preview', 'ahur-system'); ?>
                    </a>
                </div>
            </div>

            <div class="theme-author"><?php printf(__('By %s', 'ahur-system'), esc_html($theme->author)); ?></div>

            <div class="theme-description">
                <?php echo esc_html($theme->short_description); ?>
            </div>

            <div class="theme-tags">
                <span><?php _e('Tags:', 'ahur-system'); ?></span>
                <?php
                $tags = array_slice($theme->tags, 0, 3);
                echo esc_html(implode(', ', $tags));
                if (count($theme->tags) > 3) {
                    echo '...';
                }
                ?>
            </div>
        </div>
        <?php
    }

    /**
     * Format theme data for WordPress theme installation interface
     */
    private function formatThemeForWordPress($theme)
    {
        if (!is_object($theme)) {
            return null;
        }

        $formatted = array();
        error_log('Ahur System: Remote Theme Data: ' . print_r($theme, true));
        // Basic theme info
        $formatted['name'] = $theme->name ?? 'Unknown Theme';
        $formatted['slug'] = $theme->slug ?? '';
        $formatted['version'] = $theme->version ?? '1.0.0';
        $formatted['author'] = array('display_name' => $theme->author ?? 'Ahur Team');
        $formatted['author_profile'] = $theme->author_url ?? '';
        $formatted['homepage'] = $theme->homepage ?? $theme->author_url ?? '';
        $formatted['requires'] = $theme->requires ?? '5.0';
        $formatted['tested'] = $theme->tested ?? get_bloginfo('version');
        $formatted['requires_php'] = $theme->requires_php ?? '7.4';

        // Debug logging for PHP version compatibility
        error_log('Ahur System: Theme ' . $formatted['slug'] . ' requires PHP: ' . $formatted['requires_php']);
        error_log('Ahur System: Current PHP version: ' . PHP_VERSION);
        error_log('Ahur System: PHP check result: ' . (version_compare(PHP_VERSION, $formatted['requires_php'], '>=') ? 'compatible' : 'incompatible'));
        $formatted['download_link'] = $theme->downloadlink ?? $theme->download_url ?? '';

        // Description and sections
        $formatted['description'] = $theme->description ?? '';
        $formatted['short_description'] = $theme->short_description ?? $theme->description ?? '';
        $formatted['sections'] = array();
        $formatted['sections']['description'] = $theme->description ?? '';
        $formatted['sections']['installation'] = $theme->installation ?? '';
        $formatted['sections']['changelog'] = $theme->changelog ?? '';
        $formatted['sections']['faq'] = $theme->faq ?? '';
        $formatted['sections']['screenshots'] = $theme->screenshots ?? '';
        $formatted['sections']['reviews'] = $theme->reviews ?? '';
        $formatted['sections']['other_notes'] = $theme->other_notes ?? '';

        // Ratings and downloads
        $formatted['rating'] = $theme->rating ?? 0;
        $formatted['num_ratings'] = $theme->num_ratings ?? 0;
        $formatted['downloaded'] = $theme->downloaded ?? $theme->download_count ?? 0;
        $formatted['active_installs'] = $theme->active_installs ?? 0;

        // Last updated
        $formatted['last_updated'] = $theme->last_updated ?? '';

        // Tags
        if (isset($theme->tags) && is_array($theme->tags)) {
            $formatted['tags'] = $theme->tags;
        } elseif (isset($theme->tags) && is_string($theme->tags)) {
            $formatted['tags'] = explode(',', $theme->tags);
        } else {
            $formatted['tags'] = [];
        }

        // Screenshots
        if (isset($theme->screenshot_url) && $theme->screenshot_url) {
            // Convert relative URL to full HTTPS URL
            if (strpos($theme->screenshot_url, 'http') !== 0) {
                // Remove '/api' from base URL to get the main site URL
                $site_url = str_replace('/api', '', $this->api_base);
                $formatted['screenshot_url'] = $site_url . '/' . $theme->screenshot_url;
            } else {
                $formatted['screenshot_url'] = $theme->screenshot_url;
            }
            // Ensure HTTPS
            $formatted['screenshot_url'] = str_replace('http://', 'https://', $formatted['screenshot_url']);
        } else {
            $formatted['screenshot_url'] = '';
        }

        // Preview URL
        if (isset($theme->preview_url) && $theme->preview_url) {
            $formatted['preview_url'] = $theme->preview_url;
        } else {
            $formatted['preview_url'] = '';
        }

        // Icons (handle both array and object formats)
        if (isset($theme->icons)) {
            if (is_array($theme->icons)) {
                $formatted['icons'] = $theme->icons;
            } elseif (is_object($theme->icons)) {
                $formatted['icons'] = (array) $theme->icons;
            } else {
                $formatted['icons'] = [];
            }
        } else {
            $formatted['icons'] = [];
        }

        // Banners
        if (isset($theme->banners)) {
            if (is_array($theme->banners)) {
                $formatted['banners'] = $theme->banners;
            } elseif (is_object($theme->banners)) {
                $formatted['banners'] = (array) $theme->banners;
            } else {
                $formatted['banners'] = [];
            }
        } else {
            $formatted['banners'] = [];
        }

        // Add custom identifier
        $formatted['external_repository'] = true;
        $formatted['source'] = 'ahur_repository';

        return (object) $formatted;
    }

    /**
     * Handle theme installation conflicts
     */
    public function handleThemeInstallation($source, $remote_source, $upgrader, $args = [])
    {
        // Add any theme installation handling logic here
        return $source;
    }

    /**
     * Clear all caches
     */
    public function clearCache()
    {
        // Clear themes list cache
        delete_transient("ahur_themes_list_" . md5($this->api_base . '/themes/info'));

        // Clear individual theme caches
        global $wpdb;
        $cache_keys = $wpdb->get_col(
            "SELECT option_name FROM {$wpdb->options} WHERE option_name LIKE '_transient_ahur_theme_%'"
        );

        foreach ($cache_keys as $cache_key) {
            delete_transient(str_replace('_transient_', '', $cache_key));
        }
    }

    /**
     * Handle direct theme URLs by redirecting to the Ahur tab
     */
    public function handleDirectThemeUrl()
    {
        if (!isset($_GET['theme']) || !isset($_GET['page']) || $_GET['page'] !== 'theme-install') {
            return;
        }

        $theme_slug = $_GET['theme'];

        // Check if this is one of our themes
        $response_data = $this->fetchThemesList();
        if ($response_data && $response_data->success && isset($response_data->data)) {
            foreach ($response_data->data as $theme) {
                if ($theme->slug === $theme_slug) {
                    // Redirect to the Ahur tab with the theme
                    $url = admin_url('theme-install.php?tab=ahur&theme=' . $theme_slug);
                    wp_redirect($url);
                    exit;
                }
            }
        }
    }

    /**
     * Intercept HTTP requests to WordPress.org theme API
     */
    public function interceptThemeApiRequest($args, $url)
    {
        // Check if this is a theme information request to WordPress.org
        if (strpos($url, 'api.wordpress.org/themes/info/') !== false) {
            // Extract the theme slug from the URL
            if (preg_match('/themes\/info\/([^\/]+)/', $url, $matches)) {
                $theme_slug = $matches[1];

                // Check if this is one of our themes
                $response_data = $this->fetchThemesList();
                if ($response_data && $response_data->success && isset($response_data->data)) {
                    foreach ($response_data->data as $theme) {
                        if ($theme->slug === $theme_slug) {
                            error_log('Ahur System: Intercepting API request for our theme: ' . $theme_slug);
                            // Mark this request for interception
                            $args['ahur_theme_slug'] = $theme_slug;
                            break;
                        }
                    }
                }
            }
        }

        return $args;
    }

    /**
     * Intercept HTTP responses from WordPress.org theme API
     */
    public function interceptThemeApiResponse($response, $args, $url)
    {
        // Check if this was marked for interception
        if (isset($args['ahur_theme_slug'])) {
            $theme_slug = $args['ahur_theme_slug'];
            error_log('Ahur System: Intercepting API response for theme: ' . $theme_slug);

            // Fetch our theme data
            $response_data = $this->fetchThemesList();
            if ($response_data && $response_data->success && isset($response_data->data)) {
                foreach ($response_data->data as $theme) {
                    if ($theme->slug === $theme_slug) {
                        // Create the WordPress.org API response format
                        $api_response = array(
                            'name' => $theme->name,
                            'slug' => $theme->slug,
                            'version' => $theme->version,
                            'author' => $theme->author,
                            'homepage' => $theme->author_url,
                            'requires' => $theme->requires,
                            'tested' => $theme->tested,
                            'requires_php' => $theme->requires_php,
                            'download_link' => $theme->downloadlink,
                            'sections' => (array) $theme->sections,
                            'rating' => (float) $theme->rating,
                            'num_ratings' => (int) $theme->num_ratings,
                            'downloaded' => (int) $theme->downloaded,
                            'last_updated' => $theme->last_updated,
                            'tags' => (array) $theme->tags,
                            'screenshot_url' => $this->getScreenshotUrl($theme),
                            'preview_url' => $theme->preview_url,
                        );

                        // Return our response
                        return array(
                            'headers' => array('content-type' => 'application/json'),
                            'body' => json_encode($api_response),
                            'response' => array('code' => 200, 'message' => 'OK'),
                            'cookies' => array(),
                            'filename' => null,
                        );
                    }
                }
            }
        }

        return $response;
    }

    /**
     * Get screenshot URL for theme
     */
    private function getScreenshotUrl($theme)
    {
        if (isset($theme->screenshot_url) && $theme->screenshot_url) {
            if (strpos($theme->screenshot_url, 'http') !== 0) {
                $site_url = str_replace('/api', '', $this->api_base);
                return $site_url . '/' . $theme->screenshot_url;
            }
            return str_replace('http://', 'https://', $theme->screenshot_url);
        }
        return '';
    }

    /**
     * Remove directory recursively
     */
    private function removeDirectory($dir)
    {
        if (!is_dir($dir)) {
            return true;
        }

        $files = array_diff(scandir($dir), ['.', '..']);

        foreach ($files as $file) {
            $path = $dir . '/' . $file;
            if (is_dir($path)) {
                $this->removeDirectory($path);
            } else {
                unlink($path);
            }
        }

        return rmdir($dir);
    }
}
