<?php
/**
 * System Update Controller
 *
 * Main controller that manages both plugin and theme repository operations
 *
 * @package AhurSystem
 * @author Ahur Team
 * @version 0.0.9
 */

// Prevent direct access
if (!defined("ABSPATH")) {
    exit();
}

// Load repository classes
require_once AHUR_SYSTEM_PLUGIN_DIR . 'includes/repository/plugins.php';
require_once AHUR_SYSTEM_PLUGIN_DIR . 'includes/repository/themes.php';

class AhurSystemUpdate
{
    /**
     * Plugin repository manager
     */
    private $plugin_repository;

    /**
     * Theme repository manager
     */
    private $theme_repository;

    /**
     * Constructor
     */
    public function __construct()
    {
        // Don't create repositories yet - wait for init action
        add_action('init', [$this, 'initializeRepositories']);
    }

    /**
     * Initialize repositories and hooks on 'init' action
     */
    public function initializeRepositories()
    {
        $this->plugin_repository = new AhurPluginRepository();
        $this->theme_repository = new AhurThemeRepository();

        // Now initialize hooks
        $this->initHooks();
    }

    /**
     * Initialize hooks based on user preferences
     */
    public function initHooks()
    {
        // Check if updates are enabled
        if (!get_option("ahur_system_enable_updates", 1)) {
            return;
        }

        // Check what types of updates are enabled
        $enable_plugins = get_option("ahur_system_enable_plugins", 1);
        $enable_themes = get_option("ahur_system_enable_themes", 1);

        // Initialize plugin repository if enabled
        if ($enable_plugins) {
            $this->plugin_repository->initHooks();
        }

        // Initialize theme repository if enabled
        if ($enable_themes) {
            $this->theme_repository->initHooks();
        }

        // Admin menu and settings
        add_action('admin_menu', [$this, 'addAdminMenu']);

        // Add dashboard widget for repository status
        add_action('wp_dashboard_setup', [$this, 'addDashboardWidget']);
    }

    /**
     * Add admin menu items
     */
    public function addAdminMenu()
    {
        add_menu_page(
            __('Ahur Repository', 'ahur-system'),
            __('اهور', 'ahur-system'),
            'manage_options',
            'ahur-repository',
            [$this, 'renderRepositoryPage'],
            'dashicons-download',
            25
        );

        add_submenu_page(
            'ahur-repository',
            __('Plugin Repository', 'ahur-system'),
            __('Plugins', 'ahur-system'),
            'manage_options',
            'ahur-plugins',
            [$this, 'renderPluginsPage']
        );

        add_submenu_page(
            'ahur-repository',
            __('Theme Repository', 'ahur-system'),
            __('Themes', 'ahur-system'),
            'manage_options',
            'ahur-themes',
            [$this, 'renderThemesPage']
        );
    }



    /**
     * Add dashboard widget
     */
    public function addDashboardWidget()
    {
        wp_add_dashboard_widget(
            'ahur_repository_status',
            __('Ahur Repository Status', 'ahur-system'),
            [$this, 'renderDashboardWidget']
        );
    }

    /**
     * Render dashboard widget
     */
    public function renderDashboardWidget()
    {
        $plugin_status = $this->getPluginStatus();
        $theme_status = $this->getThemeStatus();

        echo '<div class="ahur-repository-status">';
        echo '<h4>' . __('Plugin Repository', 'ahur-system') . '</h4>';
        echo '<p>' . sprintf(__('Available Plugins: %d', 'ahur-system'), $plugin_status['available']) . '</p>';
        echo '<p>' . sprintf(__('Installed Plugins: %d', 'ahur-system'), $plugin_status['installed']) . '</p>';

        echo '<h4>' . __('Theme Repository', 'ahur-system') . '</h4>';
        echo '<p>' . sprintf(__('Available Themes: %d', 'ahur-system'), $theme_status['available']) . '</p>';
        echo '<p>' . sprintf(__('Installed Themes: %d', 'ahur-system'), $theme_status['installed']) . '</p>';
        echo '</div>';
    }

    /**
     * Get plugin repository status
     */
    private function getPluginStatus()
    {
        $status = [
            'available' => 0,
            'installed' => 0,
        ];

        // Get available plugins from repository
        $plugins_response = $this->plugin_repository->fetchPluginsList();
        if ($plugins_response && $plugins_response->success && isset($plugins_response->data)) {
            $status['available'] = count($plugins_response->data);
        }

        // Count installed plugins from repository
        if (function_exists('get_plugins')) {
            $all_plugins = get_plugins();
            foreach ($all_plugins as $plugin_file => $plugin_data) {
                // Check if this is one of our repository plugins
                // You might want to add a custom header or check against the repository list
            }
        }

        return $status;
    }

    /**
     * Get theme repository status
     */
    private function getThemeStatus()
    {
        $status = [
            'available' => 0,
            'installed' => 0,
        ];

        // Get available themes from repository
        $themes_response = $this->theme_repository->fetchThemesList();
        if ($themes_response && $themes_response->success && isset($themes_response->data)) {
            $status['available'] = count($themes_response->data);
        }

        // Count installed themes from repository
        $all_themes = wp_get_themes();
        foreach ($all_themes as $theme_slug => $theme) {
            // Check if this is one of our repository themes
            // You might want to add a custom header or check against the repository list
        }

        return $status;
    }

    /**
     * Render main repository page
     */
    public function renderRepositoryPage()
    {
        ?>
        <div class="wrap">
            <h1><?php _e('Ahur Repository', 'ahur-system'); ?></h1>
            
            <div class="ahur-repository-overview">
                <div class="ahur-card">
                    <h2><?php _e('Plugin Repository', 'ahur-system'); ?></h2>
                    <p><?php _e('Browse and install plugins from the Ahur repository.', 'ahur-system'); ?></p>
                    <a href="<?php echo admin_url('admin.php?page=ahur-plugins'); ?>" class="button button-primary">
                        <?php _e('Browse Plugins', 'ahur-system'); ?>
                    </a>
                </div>

                <div class="ahur-card">
                    <h2><?php _e('Theme Repository', 'ahur-system'); ?></h2>
                    <p><?php _e('Browse and install themes from the Ahur repository.', 'ahur-system'); ?></p>
                    <a href="<?php echo admin_url('admin.php?page=ahur-themes'); ?>" class="button button-primary">
                        <?php _e('Browse Themes', 'ahur-system'); ?>
                    </a>
                </div>

                <div class="ahur-card">
                    <h2><?php _e('Repository Settings', 'ahur-system'); ?></h2>
                    <p><?php _e('Configure repository connections and update preferences.', 'ahur-system'); ?></p>
                    <a href="<?php echo admin_url('admin.php?page=ahur-settings'); ?>" class="button">
                        <?php _e('Configure Settings', 'ahur-system'); ?>
                    </a>
                </div>
            </div>

            <?php $this->renderRepositoryStatus(); ?>
        </div>

        <style>
        .ahur-repository-overview {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin: 20px 0;
        }
        
        .ahur-card {
            background: #fff;
            border: 1px solid #ccd0d4;
            border-radius: 4px;
            padding: 20px;
            box-shadow: 0 1px 1px rgba(0,0,0,0.04);
        }
        
        .ahur-card h2 {
            margin-top: 0;
            color: #1e1e1e;
        }
        
        .ahur-card p {
            color: #3c434a;
            margin-bottom: 15px;
        }
        </style>
        <?php
    }

    /**
     * Render repository status
     */
    private function renderRepositoryStatus()
    {
        $plugin_status = $this->getPluginStatus();
        $theme_status = $this->getThemeStatus();
        ?>
        <div class="ahur-status-section">
            <h2><?php _e('Repository Status', 'ahur-system'); ?></h2>
            
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th><?php _e('Repository Type', 'ahur-system'); ?></th>
                        <th><?php _e('Available', 'ahur-system'); ?></th>
                        <th><?php _e('Installed', 'ahur-system'); ?></th>
                        <th><?php _e('Status', 'ahur-system'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <tr>
                        <td><strong><?php _e('Plugins', 'ahur-system'); ?></strong></td>
                        <td><?php echo $plugin_status['available']; ?></td>
                        <td><?php echo $plugin_status['installed']; ?></td>
                        <td>
                            <span class="status-active">
                                <?php _e('Active', 'ahur-system'); ?>
                            </span>
                        </td>
                    </tr>
                    <tr>
                        <td><strong><?php _e('Themes', 'ahur-system'); ?></strong></td>
                        <td><?php echo $theme_status['available']; ?></td>
                        <td><?php echo $theme_status['installed']; ?></td>
                        <td>
                            <span class="status-active">
                                <?php _e('Active', 'ahur-system'); ?>
                            </span>
                        </td>
                    </tr>
                </tbody>
            </table>
        </div>
        <?php
    }

    /**
     * Render plugins page
     */
    public function renderPluginsPage()
    {
        ?>
        <div class="wrap">
            <h1><?php _e('Ahur Plugin Repository', 'ahur-system'); ?></h1>
            
            <div class="ahur-plugin-browser">
                <p><?php _e('Browse plugins from the Ahur repository. You can also access these plugins from the WordPress plugin installation page under the "اهور" tab.', 'ahur-system'); ?></p>
                
                <p>
                    <a href="<?php echo admin_url('plugin-install.php?tab=ahur'); ?>" class="button button-primary" target="_blank">
                        <?php _e('Browse in Plugin Installer', 'ahur-system'); ?>
                    </a>
                </p>
            </div>
        </div>
        <?php
    }

    /**
     * Render themes page
     */
    public function renderThemesPage()
    {
        ?>
        <div class="wrap">
            <h1><?php _e('Ahur Theme Repository', 'ahur-system'); ?></h1>
            
            <div class="ahur-theme-browser">
                <p><?php _e('Browse themes from the Ahur repository. You can also access these themes from the WordPress theme installation page under the "اهور" tab.', 'ahur-system'); ?></p>
                
                <p>
                    <a href="<?php echo admin_url('theme-install.php?tab=ahur'); ?>" class="button button-primary" target="_blank">
                        <?php _e('Browse in Theme Installer', 'ahur-system'); ?>
                    </a>
                </p>
            </div>
        </div>
        <?php
    }

    /**
     * Render settings page
     */
    public function renderSettingsPage()
    {
        if (isset($_POST['submit'])) {
            // Save settings
            update_option('ahur_system_api_url', sanitize_url($_POST['ahur_system_api_url']));
            update_option('ahur_system_cache_time', intval($_POST['ahur_system_cache_time']));
            update_option('ahur_system_enable_updates', isset($_POST['ahur_system_enable_updates']) ? 1 : 0);
            update_option('ahur_system_enable_plugins', isset($_POST['ahur_system_enable_plugins']) ? 1 : 0);
            update_option('ahur_system_enable_themes', isset($_POST['ahur_system_enable_themes']) ? 1 : 0);
            
            echo '<div class="notice notice-success"><p>' . __('Settings saved successfully!', 'ahur-system') . '</p></div>';
        }
        ?>
        <div class="wrap">
            <h1><?php _e('Repository Settings', 'ahur-system'); ?></h1>
            
            <form method="post" action="">
                <table class="form-table" role="presentation">
                    <tbody>
                        <tr>
                            <th scope="row">
                                <label for="ahur_system_api_url"><?php _e('API URL', 'ahur-system'); ?></label>
                            </th>
                            <td>
                                <input name="ahur_system_api_url" type="url" id="ahur_system_api_url" 
                                       value="<?php echo esc_attr(get_option('ahur_system_api_url', AHUR_SYSTEM_API_BASE)); ?>" 
                                       class="regular-text">
                                <p class="description">
                                    <?php _e('The base URL for the Ahur repository API.', 'ahur-system'); ?>
                                </p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="ahur_system_cache_time"><?php _e('Cache Time (hours)', 'ahur-system'); ?></label>
                            </th>
                            <td>
                                <input name="ahur_system_cache_time" type="number" id="ahur_system_cache_time" 
                                       value="<?php echo esc_attr(get_option('ahur_system_cache_time', 6)); ?>" 
                                       class="small-text" min="1" max="24">
                                <p class="description">
                                    <?php _e('How long to cache repository data (1-24 hours).', 'ahur-system'); ?>
                                </p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row"><?php _e('Features', 'ahur-system'); ?></th>
                            <td>
                                <fieldset>
                                    <legend class="screen-reader-text">
                                        <span><?php _e('Enable Features', 'ahur-system'); ?></span>
                                    </legend>
                                    <label for="ahur_system_enable_updates">
                                        <input name="ahur_system_enable_updates" type="checkbox" 
                                               id="ahur_system_enable_updates" value="1" 
                                               <?php checked(get_option('ahur_system_enable_updates', 1)); ?>>
                                        <?php _e('Enable repository updates', 'ahur-system'); ?>
                                    </label>
                                    <br>
                                    <label for="ahur_system_enable_plugins">
                                        <input name="ahur_system_enable_plugins" type="checkbox" 
                                               id="ahur_system_enable_plugins" value="1" 
                                               <?php checked(get_option('ahur_system_enable_plugins', 1)); ?>>
                                        <?php _e('Enable plugin repository', 'ahur-system'); ?>
                                    </label>
                                    <br>
                                    <label for="ahur_system_enable_themes">
                                        <input name="ahur_system_enable_themes" type="checkbox" 
                                               id="ahur_system_enable_themes" value="1" 
                                               <?php checked(get_option('ahur_system_enable_themes', 1)); ?>>
                                        <?php _e('Enable theme repository', 'ahur-system'); ?>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                    </tbody>
                </table>
                
                <?php wp_nonce_field('ahur_system_settings'); ?>
                <p class="submit">
                    <input type="submit" name="submit" id="submit" class="button button-primary" 
                           value="<?php _e('Save Changes', 'ahur-system'); ?>">
                </p>
            </form>
        </div>
        <?php
    }
}