<?php
namespace AliKhaleghi\CurrenciesInIran;

if (!defined('ABSPATH')) {
    exit;
}

class Admin {

    private static $instance = null;
    private $page_slug = 'currencies-in-iran';

    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Format price based on display currency setting
     *
     * @param int $rate Price in IRR
     * @param int $decimal_places Number of decimal places
     * @return string Formatted price
     */
    public static function format_price($rate, $decimal_places = 0) {
        $display_currency = get_option('alikhalghi_cii_display_currency', 'IRR');

        if ($display_currency === 'IRT') {
            // Convert IRR to IRT (divide by 10)
            $toman_rate = $rate / 10;
            return number_format($toman_rate, $decimal_places) . ' IRT';
        } else {
            // Display as IRR
            return number_format($rate, $decimal_places) . ' IRR';
        }
    }

    private function __construct() {
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('wp_ajax_alikhalghi_cii_refresh', [$this, 'ajax_refresh_rates']);
        add_action('wp_ajax_alikhalghi_cii_clear_cache', [$this, 'ajax_clear_cache']);
        add_action('wp_ajax_alikhalghi_cii_clear_errors', [$this, 'ajax_clear_errors']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_scripts']);
    }

    public function add_admin_menu() {
        add_menu_page(
            __('Currencies in Iran', 'currencies-in-iran'),
            __('Currencies in Iran', 'currencies-in-iran'),
            'manage_options',
            $this->page_slug,
            [$this, 'render_admin_page'],
            'dashicons-chart-line',
            30
        );

        add_submenu_page(
            $this->page_slug,
            __('Detailed Rates', 'currencies-in-iran'),
            __('Detailed Rates', 'currencies-in-iran'),
            'manage_options',
            $this->page_slug . '-detailed',
            [$this, 'render_detailed_rates_page']
        );

        add_submenu_page(
            $this->page_slug,
            __('Sources', 'currencies-in-iran'),
            __('Sources', 'currencies-in-iran'),
            'manage_options',
            $this->page_slug . '-sources',
            [$this, 'render_sources_page']
        );

        add_submenu_page(
            $this->page_slug,
            __('Settings', 'currencies-in-iran'),
            __('Settings', 'currencies-in-iran'),
            'manage_options',
            $this->page_slug . '-settings',
            [$this, 'render_settings_page']
        );
    }

    public function register_settings() {
        register_setting('alikhalghi_cii_settings', 'alikhalghi_cii_cache_duration');
        register_setting('alikhalghi_cii_settings', 'alikhalghi_cii_show_average');
        register_setting('alikhalghi_cii_settings', 'alikhalghi_cii_decimal_places');
        register_setting('alikhalghi_cii_settings', 'alikhalghi_cii_display_currency');
    }

    public function enqueue_admin_scripts($hook) {
        if ('toplevel_page_currencies-in-iran' === $hook || 'currencies-in-iran_page_currencies-in-iran-sources' === $hook) {
            wp_enqueue_style(
                'alikhalghi-cii-admin',
                ALIKHALEGHI_CII_PLUGIN_URL . 'assets/css/admin.css',
                [],
                ALIKHALEGHI_CII_VERSION
            );

            wp_enqueue_script(
                'alikhalghi-cii-admin',
                ALIKHALEGHI_CII_PLUGIN_URL . 'assets/js/admin.js',
                ['jquery'],
                ALIKHALEGHI_CII_VERSION,
                true
            );

            wp_localize_script('alikhalghi-cii-admin', 'alikhalghi_cii_admin', [
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('alikhalghi_cii_nonce'),
                'refresh_text' => __('Refreshing rates...', 'currencies-in-iran'),
                'clear_text' => __('Clearing cache...', 'currencies-in-iran'),
                'clear_errors_text' => __('Clearing errors...', 'currencies-in-iran'),
                'success_text' => __('Success!', 'currencies-in-iran'),
                'error_text' => __('Error occurred', 'currencies-in-iran')
            ]);
        }
    }

    public function render_admin_page() {
        $cache_manager = \AliKhaleghi\CurrenciesInIran\CacheManager::get_instance();
        $source_manager = \AliKhaleghi\CurrenciesInIran\Sources\SourceManager::get_instance();

        $rates = $cache_manager->get_rates();
        $averages = [];
        $last_update = $cache_manager->get_last_update();
        $cache_status = $cache_manager->get_cache_status();
        $error_log = $cache_manager->get_error_log();
        $available_sources = $source_manager->get_available_sources();

        // Calculate averages for all currencies
        foreach ($rates as $currency => $currency_rates) {
            $averages[$currency] = $cache_manager->get_average_rate($currency);
        }

        // Calculate table data
        $table_data = $this->prepare_table_data($rates, $averages);

        ?>
        <div class="wrap">
            <div class="alikhalghi-cii-header">
                <h1><?php _e('Currencies in Iran Dashboard', 'currencies-in-iran'); ?></h1>
                <div class="alikhalghi-cii-notifications">
                    <button type="button" id="alikhalghi-cii-notifications-btn" class="button button-secondary">
                        <span class="dashicons dashicons-flag"></span>
                        <?php _e('Notifications', 'currencies-in-iran'); ?>
                        <?php if (!empty($error_log)): ?>
                            <span class="alikhalghi-cii-badge"><?php echo count($error_log); ?></span>
                        <?php endif; ?>
                    </button>
                </div>
            </div>

            <div class="alikhalghi-cii-dashboard">
                <div class="alikhalghi-cii-card">
                    <div class="alikhalghi-cii-card-header">
                        <h3><?php _e('Current Currency Rates', 'currencies-in-iran'); ?></h3>
                        <div class="alikhalghi-cii-cache-dropdown">
                            <button type="button" id="alikhalghi-cii-cache-toggle" class="button button-secondary">
                                <span class="dashicons dashicons-clock"></span>
                                <?php _e('Cache Status', 'currencies-in-iran'); ?>
                            </button>
                            <div id="alikhalghi-cii-cache-dropdown" class="alikhalghi-cii-dropdown-content" style="display: none;">
                                <div class="alikhalghi-cii-cache-info">
                                    <p><strong><?php _e('Last Update:', 'currencies-in-iran'); ?></strong>
                                        <?php echo $last_update ? esc_html(date('Y-m-d H:i:s', strtotime($last_update))) : __('Never', 'currencies-in-iran'); ?>
                                    </p>
                                    <p><strong><?php _e('Cache Status:', 'currencies-in-iran'); ?></strong>
                                        <?php echo $cache_status['has_cached_data'] ? __('Cached', 'currencies-in-iran') : __('No Cache', 'currencies-in-iran'); ?>
                                    </p>
                                    <?php if ($cache_status['has_cached_data']): ?>
                                        <p><strong><?php _e('Expires in:', 'currencies-in-iran'); ?></strong>
                                            <?php echo round($cache_status['time_until_expiry'] / 60); ?> <?php _e('minutes', 'currencies-in-iran'); ?>
                                        </p>
                                    <?php endif; ?>
                                </div>

                                <div class="alikhalghi-cii-actions">
                                    <button type="button" id="alikhalghi-cii-refresh" class="button button-primary">
                                        <?php _e('Refresh Rates', 'currencies-in-iran'); ?>
                                    </button>
                                    <button type="button" id="alikhalghi-cii-clear-cache" class="button">
                                        <?php _e('Clear Cache', 'currencies-in-iran'); ?>
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="alikhalghi-cii-table-container">
                        <table class="wp-list-table widefat fixed striped">
                            <thead>
                                <tr>
                                    <th scope="col"><?php _e('Currency', 'currencies-in-iran'); ?></th>
                                    <th scope="col"><?php _e('Highest', 'currencies-in-iran'); ?></th>
                                    <th scope="col"><?php _e('Lowest', 'currencies-in-iran'); ?></th>
                                    <th scope="col"><?php _e('Average', 'currencies-in-iran'); ?></th>
                                    <th scope="col"><?php _e('Sources', 'currencies-in-iran'); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($table_data as $currency => $data): ?>
                                    <tr>
                                        <td><strong><?php echo esc_html($currency); ?></strong></td>
                                        <td><?php echo self::format_price($data['highest'], 0); ?></td>
                                        <td><?php echo self::format_price($data['lowest'], 0); ?></td>
                                        <td><?php echo self::format_price($data['average'], 0); ?></td>
                                        <td><?php echo esc_html(implode(', ', $data['sources'])); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>

                <div class="alikhalghi-cii-card">
                    <h3><?php _e('Quick Actions', 'currencies-in-iran'); ?></h3>
                    <div class="alikhalghi-cii-quick-actions">
                        <a href="<?php echo admin_url('admin.php?page=' . $this->page_slug . '-detailed'); ?>" class="button button-secondary">
                            <?php _e('View Detailed Rates', 'currencies-in-iran'); ?>
                        </a>
                        <a href="<?php echo admin_url('admin.php?page=' . $this->page_slug . '-settings'); ?>" class="button button-secondary">
                            <?php _e('Plugin Settings', 'currencies-in-iran'); ?>
                        </a>
                    </div>
                </div>

                <div class="alikhalghi-cii-card">
                    <h3><?php _e('Shortcode Usage', 'currencies-in-iran'); ?></h3>
                    <p><?php _e('Use the following shortcode to display rates on your site:', 'currencies-in-iran'); ?></p>
                    <code>[currencies_in_iran]</code>
                    <p><?php _e('Options:', 'currencies-in-iran'); ?></p>
                    <ul>
                        <li><code>show_average="true"</code> - <?php _e('Show average rate', 'currencies-in-iran'); ?></li>
                        <li><code>sources="eghtesadnews,tgju"</code> - <?php _e('Show specific sources', 'currencies-in-iran'); ?></li>
                        <li><code>format="table"</code> - <?php _e('Display as table', 'currencies-in-iran'); ?></li>
                        <li><code>currency="EUR"</code> - <?php _e('Show specific currency (USD, EUR, GBP, SEK, AED)', 'currencies-in-iran'); ?></li>
                        <li><code>decimal_places="2"</code> - <?php _e('Number of decimal places to display', 'currencies-in-iran'); ?></li>
                    </ul>
                    <p><strong><?php _e('Note:', 'currencies-in-iran'); ?></strong> <?php _e('Currency display format (IRR/IRT) is controlled in plugin settings and applies to all shortcode outputs.', 'currencies-in-iran'); ?></p>
                </div>
            </div>
        </div>

        <!-- Notifications Modal -->
        <div id="alikhalghi-cii-notifications-modal" class="alikhalghi-cii-modal" style="display: none;">
            <div class="alikhalghi-cii-modal-content">
                <div class="alikhalghi-cii-modal-header">
                    <h3><?php _e('System Notifications', 'currencies-in-iran'); ?></h3>
                    <button type="button" class="alikhalghi-cii-modal-close">&times;</button>
                </div>
                <div class="alikhalghi-cii-modal-body">
                    <?php if (!empty($error_log)): ?>
                        <div class="alikhalghi-cii-error-log">
                            <?php foreach (array_reverse(array_slice($error_log, -10)) as $error): ?>
                                <div class="alikhalghi-cii-error-item">
                                    <span class="time"><?php echo esc_html(date('H:i:s', strtotime($error['timestamp']))); ?></span>
                                    <span class="message"><?php echo esc_html($error['message']); ?></span>
                                </div>
                            <?php endforeach; ?>
                        </div>
                        <div class="alikhalghi-cii-modal-actions">
                            <button type="button" id="alikhalghi-cii-clear-errors" class="button button-secondary">
                                <?php _e('Clear All Errors', 'currencies-in-iran'); ?>
                            </button>
                        </div>
                    <?php else: ?>
                        <p><?php _e('No notifications at this time.', 'currencies-in-iran'); ?></p>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <div id="alikhalghi-cii-loading" style="display: none;">
            <div class="spinner is-active"></div>
            <?php _e('Loading...', 'currencies-in-iran'); ?>
        </div>
        <?php
    }

    private function prepare_table_data($rates, $averages) {
        $table_data = [];

        foreach ($rates as $currency => $currency_rates) {
            if (!empty($currency_rates)) {
                $rate_values = [];
                $sources = [];

                foreach ($currency_rates as $source => $rate) {
                    $rate_values[] = $rate['rate'];
                    $sources[] = $rate['source'];
                }

                $average_rate = isset($averages[$currency]) ? $averages[$currency]['rate'] : round(array_sum($rate_values) / count($rate_values));

                $table_data[$currency] = [
                    'highest' => max($rate_values),
                    'lowest' => min($rate_values),
                    'average' => $average_rate,
                    'sources' => $sources
                ];
            }
        }

        return $table_data;
    }

    public function render_settings_page() {
        ?>
        <div class="wrap">
            <h1><?php _e('Currencies in Iran Settings', 'currencies-in-iran'); ?></h1>

            <form method="post" action="options.php">
                <?php
                settings_fields('alikhalghi_cii_settings');
                do_settings_sections('alikhalghi_cii_settings');
                ?>

                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="alikhalghi_cii_cache_duration"><?php _e('Cache Duration', 'currencies-in-iran'); ?></label>
                        </th>
                        <td>
                            <input type="number" id="alikhalghi_cii_cache_duration"
                                   name="alikhalghi_cii_cache_duration"
                                   value="<?php echo esc_attr(get_option('alikhalghi_cii_cache_duration', 15)); ?>"
                                   min="5" max="60" step="5">
                            <span class="description"><?php _e('minutes (minimum 5, maximum 60)', 'currencies-in-iran'); ?></span>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="alikhalghi_cii_show_average"><?php _e('Show Average Rate', 'currencies-in-iran'); ?></label>
                        </th>
                        <td>
                            <input type="checkbox" id="alikhalghi_cii_show_average"
                                   name="alikhalghi_cii_show_average"
                                   value="1" <?php checked(get_option('alikhalghi_cii_show_average', true)); ?>>
                            <span class="description"><?php _e('Display average rate from all sources', 'currencies-in-iran'); ?></span>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="alikhalghi_cii_decimal_places"><?php _e('Decimal Places', 'currencies-in-iran'); ?></label>
                        </th>
                        <td>
                            <input type="number" id="alikhalghi_cii_decimal_places"
                                   name="alikhalghi_cii_decimal_places"
                                   value="<?php echo esc_attr(get_option('alikhalghi_cii_decimal_places', 0)); ?>"
                                   min="0" max="2" step="1">
                            <span class="description"><?php _e('Number of decimal places to display', 'currencies-in-iran'); ?></span>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <?php _e('Display Prices In', 'currencies-in-iran'); ?>
                        </th>
                        <td>
                            <fieldset>
                                <label>
                                    <input type="radio" name="alikhalghi_cii_display_currency"
                                           value="IRR" <?php checked(get_option('alikhalghi_cii_display_currency', 'IRR'), 'IRR'); ?>>
                                    <?php _e('IRR (Iranian Rial)', 'currencies-in-iran'); ?>
                                </label>
                                <br>
                                <label>
                                    <input type="radio" name="alikhalghi_cii_display_currency"
                                           value="IRT" <?php checked(get_option('alikhalghi_cii_display_currency', 'IRR'), 'IRT'); ?>>
                                    <?php _e('IRT (Iranian Toman)', 'currencies-in-iran'); ?>
                                </label>
                                <br>
                                <span class="description"><?php _e('Choose currency format for displaying prices (IRR = Rial, IRT = Toman)', 'currencies-in-iran'); ?></span>
                            </fieldset>
                        </td>
                    </tr>
                </table>

                <?php submit_button(); ?>
            </form>
        </div>
        <?php
    }

    public function ajax_refresh_rates() {
        check_ajax_referer('alikhalghi_cii_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions', 'currencies-in-iran'));
        }

        $cache_manager = \AliKhaleghi\CurrenciesInIran\CacheManager::get_instance();
        $result = $cache_manager->force_refresh();

        if (is_wp_error($result)) {
            wp_send_json_error([
                'message' => $result->get_error_message()
            ]);
        }

        wp_send_json_success([
            'rates' => $result,
            'last_update' => $cache_manager->get_last_update()
        ]);
    }

    public function ajax_clear_cache() {
        check_ajax_referer('alikhalghi_cii_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions', 'currencies-in-iran'));
        }

        $cache_manager = \AliKhaleghi\CurrenciesInIran\CacheManager::get_instance();
        $result = $cache_manager->clear_cache();

        if ($result) {
            wp_send_json_success([
                'message' => __('Cache cleared successfully', 'currencies-in-iran')
            ]);
        } else {
            wp_send_json_error([
                'message' => __('Failed to clear cache', 'currencies-in-iran')
            ]);
        }
    }

    public function ajax_clear_errors() {
        check_ajax_referer('alikhalghi_cii_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions', 'currencies-in-iran'));
        }

        $cache_manager = \AliKhaleghi\CurrenciesInIran\CacheManager::get_instance();
        $result = $cache_manager->clear_error_log();

        if ($result) {
            wp_send_json_success([
                'message' => __('Errors cleared successfully', 'currencies-in-iran')
            ]);
        } else {
            wp_send_json_error([
                'message' => __('Failed to clear errors', 'currencies-in-iran')
            ]);
        }
    }

    public function render_sources_page() {
        include ALIKHALEGHI_CII_PLUGIN_DIR . 'includes/views/sources.php';
    }

    public function render_detailed_rates_page() {
        $cache_manager = \AliKhaleghi\CurrenciesInIran\CacheManager::get_instance();
        $source_manager = \AliKhaleghi\CurrenciesInIran\Sources\SourceManager::get_instance();

        $rates = $cache_manager->get_rates();
        $last_update = $cache_manager->get_last_update();
        $available_sources = $source_manager->get_available_sources();

        ?>
        <div class="wrap">
            <h1><?php _e('Detailed Currency Rates', 'currencies-in-iran'); ?></h1>

            <div class="alikhalghi-cii-detailed-dashboard">
                <?php if (!empty($rates)): ?>
                    <?php foreach ($rates as $currency => $currency_rates): ?>
                        <div class="alikhalghi-cii-card">
                            <div class="alikhalghi-cii-card-header">
                                <h3><?php echo esc_html($currency); ?> <?php _e('Rates', 'currencies-in-iran'); ?></h3>
                                <span class="alikhalghi-cii-currency-count">
                                    <?php echo sprintf(__('%d sources', 'currencies-in-iran'), count($currency_rates)); ?>
                                </span>
                            </div>

                            <div class="alikhalghi-cii-detailed-rates-grid">
                                <?php foreach ($currency_rates as $source => $rate): ?>
                                    <div class="alikhalghi-cii-detailed-rate-card">
                                        <div class="alikhalghi-cii-rate-header">
                                            <h4><?php echo esc_html($rate['source']); ?></h4>
                                            <span class="alikhalghi-cii-rate-time">
                                                <?php echo esc_html(date('H:i', strtotime($rate['timestamp']))); ?>
                                            </span>
                                        </div>
                                        <div class="alikhalghi-cii-rate-value">
                                            <?php echo self::format_price($rate['rate'], 0); ?>
                                        </div>
                                        <div class="alikhalghi-cii-rate-details">
                                            <span class="alikhalghi-cii-rate-date">
                                                <?php echo esc_html(date('Y-m-d', strtotime($rate['timestamp']))); ?>
                                            </span>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php else: ?>
                    <div class="alikhalghi-cii-card">
                        <p><?php _e('No rates available. Please try refreshing from the main dashboard.', 'currencies-in-iran'); ?></p>
                        <p>
                            <a href="<?php echo admin_url('admin.php?page=' . $this->page_slug); ?>" class="button button-primary">
                                <?php _e('Go to Dashboard', 'currencies-in-iran'); ?>
                            </a>
                        </p>
                    </div>
                <?php endif; ?>

                <div class="alikhalghi-cii-card">
                    <h3><?php _e('Available Sources', 'currencies-in-iran'); ?></h3>
                    <div class="alikhalghi-cii-sources-grid">
                        <?php foreach ($available_sources as $currency => $sources): ?>
                            <div class="alikhalghi-cii-source-group">
                                <h4><?php echo esc_html($currency); ?></h4>
                                <ul>
                                    <?php foreach ($sources as $source): ?>
                                        <li><?php echo esc_html($source); ?></li>
                                    <?php endforeach; ?>
                                </ul>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>

                <?php if ($last_update): ?>
                    <div class="alikhalghi-cii-card">
                        <p class="alikhalghi-cii-last-update">
                            <strong><?php _e('Last Update:', 'currencies-in-iran'); ?></strong>
                            <?php echo esc_html(date('Y-m-d H:i:s', strtotime($last_update))); ?>
                        </p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }
}
