<?php
namespace AliKhaleghi\CurrenciesInIran;

if (!defined('ABSPATH')) {
    exit;
}

class CacheManager {
    
    private static $instance = null;
    private $cache_key = 'alikhalghi_cii_rates';
    private $last_update_key = 'alikhalghi_cii_last_update';
    private $error_log_key = 'alikhalghi_cii_error_log';
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_filter('cron_schedules', [$this, 'add_fifteen_minute_schedule']);
    }
    
    public function add_fifteen_minute_schedule($schedules) {
        if (!isset($schedules['fifteen_minutes'])) {
            $schedules['fifteen_minutes'] = [
                'interval' => 15 * MINUTE_IN_SECONDS,
                'display' => __('Every 15 minutes', 'currencies-in-iran')
            ];
        }
        return $schedules;
    }
    
    public function get_rates() {
        $cached_rates = get_transient($this->cache_key);
        
        if (false !== $cached_rates) {
            return $cached_rates;
        }
        
        $source_manager = \AliKhaleghi\CurrenciesInIran\Sources\SourceManager::get_instance();
        $rates = $source_manager->get_all_rates();
        
        if (!empty($rates)) {
            $this->set_rates($rates);
            return $rates;
        }
        
        $this->log_error('Failed to fetch rates: no data returned');
        return [];
    }
    
    public function set_rates($rates) {
        if (empty($rates)) {
            return false;
        }
        
        $validated_rates = [];
        foreach ($rates as $currency => $currency_rates) {
            foreach ($currency_rates as $source => $rate) {
                if ($this->validate_rate_data($rate)) {
                    $validated_rates[$currency][$source] = $rate;
                } else {
                    $this->log_error("Invalid rate data from {$source} ({$currency})");
                }
            }
        }
        
        if (empty($validated_rates)) {
            return false;
        }
        
        $success = set_transient(
            $this->cache_key,
            $validated_rates,
            ALIKHALEGHI_CII_CACHE_EXPIRY
        );
        
        if ($success) {
            update_option($this->last_update_key, current_time('mysql'));
        }
        
        return $success;
    }
    
    public function get_last_update() {
        return get_option($this->last_update_key, false);
    }
    
    public function clear_cache() {
        delete_transient($this->cache_key);
        delete_option($this->last_update_key);
        return true;
    }
    
    public function force_refresh() {
        $this->clear_cache();
        
        $source_manager = \AliKhaleghi\CurrenciesInIran\Sources\SourceManager::get_instance();
        $rates = $source_manager->get_all_rates();
        
        if (empty($rates)) {
            $this->log_error('Force refresh failed: no data returned');
            return new \WP_Error('refresh_failed', 'No data returned from sources');
        }
        
        $this->set_rates($rates);
        return $rates;
    }
    
    public function get_average_rate($currency = 'USD') {
        $rates = $this->get_rates();
        
        if (!isset($rates[$currency]) || empty($rates[$currency])) {
            return false;
        }
        
        $total = 0;
        $count = 0;
        
        foreach ($rates[$currency] as $rate) {
            if (isset($rate['rate']) && is_numeric($rate['rate'])) {
                $total += $rate['rate'];
                $count++;
            }
        }
        
        if ($count === 0) {
            return false;
        }
        
        return [
            'source' => 'Average',
            'currency' => $currency,
            'rate' => round($total / $count),
            'timestamp' => current_time('mysql'),
            'sources_count' => $count
        ];
    }
    
    public function get_cache_status() {
        $cached = get_transient($this->cache_key);
        $last_update = $this->get_last_update();
        
        return [
            'has_cached_data' => false !== $cached,
            'last_update' => $last_update,
            'cache_expiry' => ALIKHALEGHI_CII_CACHE_EXPIRY,
            'time_until_expiry' => false !== $cached ? 
                ALIKHALEGHI_CII_CACHE_EXPIRY - (time() - strtotime($last_update)) : 
                0
        ];
    }
    
    public function log_error($message) {
        $error_log = get_option($this->error_log_key, []);
        
        $error_log[] = [
            'message' => $message,
            'timestamp' => current_time('mysql'),
            'url' => $_SERVER['REQUEST_URI'] ?? 'unknown'
        ];
        
        if (count($error_log) > 50) {
            $error_log = array_slice($error_log, -50);
        }
        
        update_option($this->error_log_key, $error_log);
        
        error_log('Currencies in Iran: ' . $message);
    }
    
    public function get_error_log() {
        return get_option($this->error_log_key, []);
    }
    
    public function clear_error_log() {
        return delete_option($this->error_log_key);
    }
    
    public function export_rates() {
        $rates = $this->get_rates();
        $average = $this->get_average_rate();
        
        $export_data = [
            'timestamp' => current_time('mysql'),
            'rates' => $rates,
            'average' => $average,
            'last_update' => $this->get_last_update(),
            'cache_status' => $this->get_cache_status()
        ];
        
        return $export_data;
    }
    
    private function validate_rate_data($rate) {
        if (!is_array($rate)) {
            return false;
        }
        
        $required_fields = ['source', 'currency', 'rate', 'timestamp'];
        foreach ($required_fields as $field) {
            if (!isset($rate[$field])) {
                return false;
            }
        }
        
        if (!is_numeric($rate['rate'])) {
            return false;
        }
        
        // Currency-specific validation ranges
        $currency = strtoupper($rate['currency']);
        switch ($currency) {
            case 'USD':
                return $rate['rate'] >= 50000 && $rate['rate'] <= 3000000;
            case 'EUR':
                return $rate['rate'] >= 50000 && $rate['rate'] <= 5000000;
            case 'GBP':
                return $rate['rate'] >= 60000 && $rate['rate'] <= 6000000;
            case 'SEK':
                return $rate['rate'] >= 10000 && $rate['rate'] <= 500000;
            default:
                // Generic range for unknown currencies
                return $rate['rate'] >= 1000 && $rate['rate'] <= 10000000;
        }
    }
}