<?php
namespace AliKhaleghi\CurrenciesInIran;

if (!defined('ABSPATH')) {
    exit;
}

class Shortcode {
    
    private static $instance = null;
    private $cache_manager;
    private $source_manager;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    public function __construct() {
        $this->cache_manager = \AliKhaleghi\CurrenciesInIran\CacheManager::get_instance();
        $this->source_manager = \AliKhaleghi\CurrenciesInIran\Sources\SourceManager::get_instance();
        
        add_shortcode('currencies_in_iran', [$this, 'render_shortcode']);
    }
    
    /**
     * Format price based on display currency setting
     * 
     * @param int $rate Price in IRR
     * @param int $decimal_places Number of decimal places
     * @return string Formatted price
     */
    private function format_price($rate, $decimal_places = 0) {
        $display_currency = get_option('alikhalghi_cii_display_currency', 'IRR');
        
        if ($display_currency === 'IRT') {
            // Convert IRR to IRT (divide by 10)
            $toman_rate = $rate / 10;
            return number_format($toman_rate, $decimal_places) . ' IRT';
        } else {
            // Display as IRR
            return number_format($rate, $decimal_places) . ' IRR';
        }
    }
    
    public function render_shortcode($atts) {
        $atts = shortcode_atts([
            'show_average' => get_option('alikhalghi_cii_show_average', true),
            'sources' => '',
            'currency' => 'USD',
            'format' => 'list',
            'show_timestamp' => false,
            'show_source' => true,
            'decimal_places' => get_option('alikhalghi_cii_decimal_places', 0),
        ], $atts, 'currencies_in_iran');
        
        $cache_manager = \AliKhaleghi\CurrenciesInIran\CacheManager::get_instance();
        $rates = $cache_manager->get_rates();
        
        if (empty($rates) || !isset($rates[$atts['currency']])) {
            return '<p class="cii-no-data">' . __('Currency rates are currently unavailable.', 'currencies-in-iran') . '</p>';
        }
        
        $currency_rates = $rates[$atts['currency']];
        $filtered_rates = $this->filter_rates($currency_rates, $atts['sources']);
        $average_rate = $atts['show_average'] ? $cache_manager->get_average_rate($atts['currency']) : null;
        
        $output = '<div class="currencies-in-iran ' . esc_attr('cii-format-' . $atts['format']) . '">';
        
        if ($atts['format'] === 'table') {
            $output .= $this->render_table_format($filtered_rates, $average_rate, $atts);
        } else {
            $output .= $this->render_list_format($filtered_rates, $average_rate, $atts);
        }
        
        $last_update = $cache_manager->get_last_update();
        if ($last_update && $atts['show_timestamp']) {
            $output .= '<p class="cii-last-update">' . 
                sprintf(__('Last updated: %s', 'currencies-in-iran'), 
                    esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($last_update)))) . 
                '</p>';
        }
        
        $output .= '</div>';
        
        return $output;
    }
    
    private function filter_rates($rates, $sources_filter) {
        if (empty($sources_filter)) {
            return $rates;
        }
        
        $allowed_sources = array_map('trim', explode(',', $sources_filter));
        $filtered_rates = [];
        
        foreach ($rates as $source => $rate) {
            if (in_array($source, $allowed_sources)) {
                $filtered_rates[$source] = $rate;
            }
        }
        
        return $filtered_rates;
    }
    
    private function render_list_format($rates, $average_rate, $atts) {
        $output = '<ul class="cii-rates-list">';
        
        foreach ($rates as $source => $rate) {
            $output .= $this->render_rate_item($rate, $atts);
        }
        
        if ($average_rate) {
            $output .= $this->render_rate_item($average_rate, $atts, true);
        }
        
        $output .= '</ul>';
        
        return $output;
    }
    
    private function render_table_format($rates, $average_rate, $atts) {
        $output = '<table class="cii-rates-table">';
        $output .= '<thead><tr>';
        
        if ($atts['show_source']) {
            $output .= '<th>' . __('Source', 'currencies-in-iran') . '</th>';
        }
        
        $output .= '<th>' . __('Rate (IRR)', 'currencies-in-iran') . '</th>';
        
        if ($atts['show_timestamp']) {
            $output .= '<th>' . __('Time', 'currencies-in-iran') . '</th>';
        }
        
        $output .= '</tr></thead><tbody>';
        
        foreach ($rates as $source => $rate) {
            $output .= $this->render_table_row($rate, $atts);
        }
        
        if ($average_rate) {
            $output .= $this->render_table_row($average_rate, $atts, true);
        }
        
        $output .= '</tbody></table>';
        
        return $output;
    }
    
    private function render_rate_item($rate, $atts, $is_average = false) {
        $class = $is_average ? 'cii-rate-average' : 'cii-rate-item';
        $source_label = $is_average ? __('Average', 'currencies-in-iran') : $rate['source'];
        
        $output = '<li class="' . esc_attr($class) . '">';
        
        if ($atts['show_source']) {
            $output .= '<span class="cii-source">' . esc_html($source_label) . '</span>';
        }
        
        $output .= '<span class="cii-rate">' . 
            $this->format_price($rate['rate'], (int) $atts['decimal_places']) . '</span>';
        
        if ($atts['show_timestamp'] && !$is_average) {
            $output .= '<span class="cii-time">' . 
                esc_html(date('H:i', strtotime($rate['timestamp']))) . '</span>';
        }
        
        $output .= '</li>';
        
        return $output;
    }
    
    private function render_table_row($rate, $atts, $is_average = false) {
        $class = $is_average ? 'cii-average-row' : '';
        $source_label = $is_average ? __('Average', 'currencies-in-iran') : $rate['source'];
        
        $output = '<tr class="' . esc_attr($class) . '">';
        
        if ($atts['show_source']) {
            $output .= '<td class="cii-source">' . esc_html($source_label) . '</td>';
        }
        
        $output .= '<td class="cii-rate">' . 
            $this->format_price($rate['rate'], (int) $atts['decimal_places']) . '</td>';
        
        if ($atts['show_timestamp'] && !$is_average) {
            $output .= '<td class="cii-time">' . 
                esc_html(date('H:i', strtotime($rate['timestamp']))) . '</td>';
        } elseif ($atts['show_timestamp'] && $is_average) {
            $output .= '<td class="cii-time">-</td>';
        }
        
        $output .= '</tr>';
        
        return $output;
    }
}