<?php
namespace AliKhaleghi\CurrenciesInIran\Sources;

if (!defined('ABSPATH')) {
    exit;
}

class SourceManager {
    
    private static $instance = null;
    private $sources = [];
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        $this->load_sources();
    }
    
    private function load_sources() {
        require_once ALIKHALEGHI_CII_PLUGIN_DIR . 'includes/sources/class-source-loader.php';
        require_once ALIKHALEGHI_CII_PLUGIN_DIR . 'includes/sources/class-source-validator.php';
        
        \AliKhaleghi\CurrenciesInIran\Sources\SourceLoader::load_interface();
        
        $this->load_usd_sources();
        $this->load_eur_sources();
        $this->load_gbp_sources();
        $this->load_sek_sources();
        $this->load_aed_sources();
        
        // Validate sources in debug mode
        if (defined('WP_DEBUG') && WP_DEBUG) {
            \AliKhaleghi\CurrenciesInIran\Sources\SourceValidator::log_validation_results();
        }
        
        do_action('alikhalghi_cii_load_sources', $this);
    }
    
    private function load_usd_sources() {
        $usd_sources = [
            'eghtesadnews' => 'EghtesadNews',
            'tgju' => 'Tgju',
        ];

        $enabled_sources = get_option('alikhalghi_cii_enabled_sources', []);

        // If no settings, enable all by default
        if (empty($enabled_sources)) {
            foreach ($usd_sources as $slug => $class_name) {
                $enabled_sources['USD'][$slug] = true;
            }
        }

        foreach ($usd_sources as $slug => $class_name) {
            if (!isset($enabled_sources['USD'][$slug]) || !$enabled_sources['USD'][$slug]) {
                continue;
            }

            if (\AliKhaleghi\CurrenciesInIran\Sources\SourceLoader::load_source('usd', $slug, $class_name)) {
                $class = "AliKhaleghi\\CurrenciesInIran\\Sources\\USD\\{$class_name}";

                if (class_exists($class)) {
                    $source = new $class();

                    if ($source instanceof \AliKhaleghi\CurrenciesInIran\Sources\CurrencyInterface) {
                        $this->sources['USD'][$slug] = $source;
                    }
                }
            }
        }
    }
    
    private function load_eur_sources() {
        $eur_sources = [
            'tgju' => 'Tgju',
        ];

        $enabled_sources = get_option('alikhalghi_cii_enabled_sources', []);

        // If no settings, enable all by default
        if (empty($enabled_sources)) {
            foreach ($eur_sources as $slug => $class_name) {
                $enabled_sources['EUR'][$slug] = true;
            }
        }

        foreach ($eur_sources as $slug => $class_name) {
            if (!isset($enabled_sources['EUR'][$slug]) || !$enabled_sources['EUR'][$slug]) {
                continue;
            }

            if (\AliKhaleghi\CurrenciesInIran\Sources\SourceLoader::load_source('eur', $slug, $class_name)) {
                $class = "AliKhaleghi\\CurrenciesInIran\\Sources\\EUR\\{$class_name}";

                if (class_exists($class)) {
                    $source = new $class();

                    if ($source instanceof \AliKhaleghi\CurrenciesInIran\Sources\CurrencyInterface) {
                        $this->sources['EUR'][$slug] = $source;
                    }
                }
            }
        }
    }
    
    private function load_gbp_sources() {
        $gbp_sources = [
            'tgju' => 'Tgju',
        ];

        $enabled_sources = get_option('alikhalghi_cii_enabled_sources', []);

        // If no settings, enable all by default
        if (empty($enabled_sources)) {
            foreach ($gbp_sources as $slug => $class_name) {
                $enabled_sources['GBP'][$slug] = true;
            }
        }

        foreach ($gbp_sources as $slug => $class_name) {
            if (!isset($enabled_sources['GBP'][$slug]) || !$enabled_sources['GBP'][$slug]) {
                continue;
            }

            if (\AliKhaleghi\CurrenciesInIran\Sources\SourceLoader::load_source('gbp', $slug, $class_name)) {
                $class = "AliKhaleghi\\CurrenciesInIran\\Sources\\GBP\\{$class_name}";

                if (class_exists($class)) {
                    $source = new $class();

                    if ($source instanceof \AliKhaleghi\CurrenciesInIran\Sources\CurrencyInterface) {
                        $this->sources['GBP'][$slug] = $source;
                    }
                }
            }
        }
    }
    
    private function load_sek_sources() {
        $sek_sources = [
            'tgju' => 'Tgju',
        ];

        $enabled_sources = get_option('alikhalghi_cii_enabled_sources', []);

        // If no settings, enable all by default
        if (empty($enabled_sources)) {
            foreach ($sek_sources as $slug => $class_name) {
                $enabled_sources['SEK'][$slug] = true;
            }
        }

        foreach ($sek_sources as $slug => $class_name) {
            if (!isset($enabled_sources['SEK'][$slug]) || !$enabled_sources['SEK'][$slug]) {
                continue;
            }

            if (\AliKhaleghi\CurrenciesInIran\Sources\SourceLoader::load_source('sek', $slug, $class_name)) {
                $class = "AliKhaleghi\\CurrenciesInIran\\Sources\\SEK\\{$class_name}";

                if (class_exists($class)) {
                    $source = new $class();

                    if ($source instanceof \AliKhaleghi\CurrenciesInIran\Sources\CurrencyInterface) {
                        $this->sources['SEK'][$slug] = $source;
                    }
                }
            }
        }
    }

    private function load_aed_sources() {
        $aed_sources = [
            'alanchand' => 'Alanchand',
        ];

        $enabled_sources = get_option('alikhalghi_cii_enabled_sources', []);

        // If no settings, enable all by default
        if (empty($enabled_sources)) {
            foreach ($aed_sources as $slug => $class_name) {
                $enabled_sources['AED'][$slug] = true;
            }
        }

        foreach ($aed_sources as $slug => $class_name) {
            if (!isset($enabled_sources['AED'][$slug]) || !$enabled_sources['AED'][$slug]) {
                continue;
            }

            if (\AliKhaleghi\CurrenciesInIran\Sources\SourceLoader::load_source('aed', $slug, $class_name)) {
                $class = "AliKhaleghi\\CurrenciesInIran\\Sources\\AED\\{$class_name}";

                if (class_exists($class)) {
                    $source = new $class();

                    if ($source instanceof \AliKhaleghi\CurrenciesInIran\Sources\CurrencyInterface) {
                        $this->sources['AED'][$slug] = $source;
                    }
                }
            }
        }
    }
    
    public function get_sources($currency = null) {
        if ($currency) {
            return isset($this->sources[$currency]) ? $this->sources[$currency] : [];
        }
        
        return $this->sources;
    }
    
    public function get_source($currency, $slug) {
        if (isset($this->sources[$currency][$slug])) {
            return $this->sources[$currency][$slug];
        }
        
        return null;
    }
    
    public function get_all_rates() {
        $rates = [];
        $errors = [];
        
        foreach ($this->sources as $currency => $currency_sources) {
            foreach ($currency_sources as $slug => $source) {
                try {
                    $result = $source->scrape_rate();
                    
                    if (is_wp_error($result)) {
                        $errors[] = [
                            'source' => $source->get_source_name(),
                            'currency' => $currency,
                            'error' => $result->get_error_message()
                        ];
                        continue;
                    }
                    
                    $rates[$currency][$slug] = $result;
                    
                } catch (Exception $e) {
                    $errors[] = [
                        'source' => $source->get_source_name(),
                        'currency' => $currency,
                        'error' => $e->getMessage()
                    ];
                }
            }
        }
        
        if (!empty($errors)) {
            error_log('Currencies in Iran - Scraping errors: ' . print_r($errors, true));
        }
        
        return $rates;
    }
    
    public function get_currency_rates($currency) {
        if (!isset($this->sources[$currency])) {
            return new WP_Error('invalid_currency', "No sources available for currency: {$currency}");
        }
        
        $rates = [];
        $errors = [];
        
        foreach ($this->sources[$currency] as $slug => $source) {
            try {
                $result = $source->scrape_rate();
                
                if (is_wp_error($result)) {
                    $errors[] = [
                        'source' => $source->get_source_name(),
                        'error' => $result->get_error_message()
                    ];
                    continue;
                }
                
                $rates[$slug] = $result;
                
            } catch (Exception $e) {
                $errors[] = [
                    'source' => $source->get_source_name(),
                    'error' => $e->getMessage()
                ];
            }
        }
        
        if (!empty($errors)) {
            error_log("Currencies in Iran - {$currency} scraping errors: " . print_r($errors, true));
        }
        
        return $rates;
    }
    
    public function get_single_rate($currency, $source_slug) {
        $source = $this->get_source($currency, $source_slug);
        
        if (!$source) {
            return new WP_Error('source_not_found', "Source {$source_slug} not found for currency {$currency}");
        }
        
        try {
            $result = $source->scrape_rate();
            
            if (is_wp_error($result)) {
                return $result;
            }
            
            return $result;
            
        } catch (Exception $e) {
            return new WP_Error('scraping_error', $e->getMessage());
        }
    }
    
    public function get_available_currencies() {
        return array_keys($this->sources);
    }
    
    public function get_available_sources($currency = null) {
        if ($currency === null) {
            // Return all sources for all currencies
            $all_sources = [];
            foreach ($this->sources as $currency_code => $currency_sources) {
                $all_sources[$currency_code] = [];
                foreach ($currency_sources as $slug => $source) {
                    $all_sources[$currency_code][$slug] = $source->get_source_name();
                }
            }
            return $all_sources;
        }
        
        if (!isset($this->sources[$currency])) {
            return [];
        }
        
        $sources = [];
        foreach ($this->sources[$currency] as $slug => $source) {
            $sources[$slug] = [
                'name' => $source->get_source_name(),
                'url' => $source->get_source_url(),
                'currency' => $source->get_currency_code(),
            ];
        }
        
        return $sources;
    }
    
    public function add_source($currency, $slug, \AliKhaleghi\CurrenciesInIran\Sources\CurrencyInterface $source) {
        $this->sources[$currency][$slug] = $source;
    }
    
    public function remove_source($currency, $slug) {
        if (isset($this->sources[$currency][$slug])) {
            unset($this->sources[$currency][$slug]);
            return true;
        }
        
        return false;
    }
    
    public function validate_all_sources() {
        $validation_results = [];
        
        foreach ($this->sources as $currency => $currency_sources) {
            foreach ($currency_sources as $slug => $source) {
                try {
                    $result = $source->scrape_rate();
                    
                    $validation_results[$currency][$slug] = [
                        'valid' => !is_wp_error($result),
                        'source' => $source->get_source_name(),
                        'result' => is_wp_error($result) ? $result->get_error_message() : 'Success',
                        'rate' => !is_wp_error($result) ? $result['rate'] : null,
                    ];
                    
                } catch (Exception $e) {
                    $validation_results[$currency][$slug] = [
                        'valid' => false,
                        'source' => $source->get_source_name(),
                        'result' => $e->getMessage(),
                        'rate' => null,
                    ];
                }
            }
        }
        
        return $validation_results;
    }
}