<?php
namespace AliKhaleghi\CurrenciesInIran\Sources\AED;

use AliKhaleghi\CurrenciesInIran\Sources\CurrencyInterface;

if (!defined('ABSPATH')) {
    exit;
}

class Alanchand implements CurrencyInterface {

    private $url = 'https://alanchand.com/currencies-price/aed';
    private $user_agent;

    public function __construct() {
        $this->user_agent = 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36';
    }

    public function get_currency_code() {
        return 'AED';
    }

    public function get_source_name() {
        return 'Alanchand';
    }

    public function get_source_url() {
        return $this->url;
    }

    public function get_user_agent() {
        return $this->user_agent;
    }

    public function get_request_headers() {
        return [
            'Accept' => 'text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
            'Accept-Language' => 'fa-IR,fa;q=0.8,en-US;q=0.5,en;q=0.3',
            'Accept-Encoding' => 'gzip, deflate',
            'Connection' => 'keep-alive',
            'Upgrade-Insecure-Requests' => '1',
            'Cache-Control' => 'max-age=0',
        ];
    }

    public function get_timeout() {
        return 15;
    }

    public function scrape_rate() {
        $response = wp_remote_get($this->url, [
            'timeout' => $this->get_timeout(),
            'user-agent' => $this->get_user_agent(),
            'headers' => $this->get_request_headers(),
        ]);

        if (is_wp_error($response)) {
            return new \WP_Error('http_error', 'Failed to fetch Alanchand: ' . $response->get_error_message());
        }

        $body = wp_remote_retrieve_body($response);
        if (empty($body)) {
            return new \WP_Error('empty_response', 'Empty response from Alanchand');
        }

        $rate = $this->extract_rate_from_html($body);

        if (!$rate) {
            return new \WP_Error('extraction_failed', 'Could not extract valid AED rate from Alanchand');
        }

        if (!$this->validate_rate($rate)) {
            return new \WP_Error('validation_failed', 'Extracted rate failed validation: ' . $rate);
        }

        return $this->format_rate_data($rate, current_time('mysql'));
    }

    private function extract_rate_from_html($html) {
        // Try to match the specific span with the selling price
        if (preg_match('/<span[^>]*class="fw-bold[^"]*text-success[^"]*fs-5"[^>]*>([۰-۹,]+)<\/span>/', $html, $matches)) {
            $rate = $this->clean_and_parse_rate($matches[1]);
            if ($rate) {
                return $rate;
            }
        }
        
        // Fallback: regex for "قیمت فروش درهم"
        if (preg_match('/قیمت فروش درهم.*?([۰-۹,]+)%?/s', $html, $matches)) {
            $rate = $this->clean_and_parse_rate($matches[1]);
            if ($rate) {
                return $rate;
            }
        }
        
        $dom = new \DOMDocument();
        
        libxml_use_internal_errors(true);
        $dom->loadHTML($html);
        libxml_clear_errors();
        
        $xpath = new \DOMXPath($dom);
        
        // Look for the selling price under "قیمت فروش درهم"
        $selectors = [
            '//h2[text()="قیمت فروش درهم"]/following-sibling::*[1]//text()',
            '//h2[contains(text(), "قیمت فروش درهم")]/following-sibling::*[1]//text()',
            '//div[contains(text(), "۳۴,۰۹۰")]//text()',
        ];
        
        foreach ($selectors as $selector) {
            $nodes = $xpath->query($selector);
            foreach ($nodes as $node) {
                $text = trim($node->nodeValue);
                $rate = $this->clean_and_parse_rate($text);
                if ($rate) {
                    return $rate;
                }
            }
        }
        
        return false;
    }

    private function clean_and_parse_rate($text) {
        if (empty($text)) {
            return false;
        }

        // Convert Persian digits to Latin
        $persian_digits = ['۰', '۱', '۲', '۳', '۴', '۵', '۶', '۷', '۸', '۹'];
        $latin_digits = ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9'];
        $text = str_replace($persian_digits, $latin_digits, $text);

        // Remove non-numeric except comma
        $text = preg_replace('/[^\d,]/', '', $text);
        $text = str_replace(',', '', $text);

        if (!is_numeric($text)) {
            return false;
        }

        $rate = (int) $text;

        // Reasonable range for AED to IRR
        if ($rate < 20000 || $rate > 50000) {
            return false;
        }

        return $rate;
    }

    public function validate_rate($rate) {
        if (!is_numeric($rate)) {
            return false;
        }

        $rate = (float) $rate;

        // Allow range: 20,000 - 50,000 IRR for AED
        if ($rate < 20000 || $rate > 50000) {
            return false;
        }

        return true;
    }

    public function format_rate_data($rate, $timestamp) {
        return [
            'source' => $this->get_source_name(),
            'currency' => $this->get_currency_code(),
            'rate' => $rate,
            'timestamp' => $timestamp,
            'url' => $this->get_source_url()
        ];
    }
}