<?php
namespace AliKhaleghi\CurrenciesInIran\Sources\SEK;

use AliKhaleghi\CurrenciesInIran\Sources\CurrencyInterface;

if (!defined('ABSPATH')) {
    exit;
}

class Tgju implements CurrencyInterface {
    
    private $url = 'https://www.tgju.org/profile/price_sek';
    private $user_agent;
    
    public function __construct() {
        $this->user_agent = 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36';
    }
    
    public function get_currency_code() {
        return 'SEK';
    }
    
    public function get_source_name() {
        return 'Tgju';
    }
    
    public function get_source_url() {
        return $this->url;
    }
    
    public function get_user_agent() {
        return $this->user_agent;
    }
    
    public function get_request_headers() {
        return [
            'Accept' => 'text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
            'Accept-Language' => 'fa-IR,fa;q=0.8,en-US;q=0.5,en;q=0.3',
            'Accept-Encoding' => 'gzip, deflate',
            'Connection' => 'keep-alive',
            'Upgrade-Insecure-Requests' => '1',
            'Cache-Control' => 'max-age=0',
            'X-Requested-With' => 'XMLHttpRequest',
            'Referer' => 'https://www.tgju.org/',
        ];
    }
    
    public function get_timeout() {
        return 15;
    }
    
    public function scrape_rate() {
        $response = wp_remote_get($this->url, [
            'timeout' => $this->get_timeout(),
            'user-agent' => $this->get_user_agent(),
            'headers' => $this->get_request_headers(),
        ]);
        
        if (is_wp_error($response)) {
            return new \WP_Error('http_error', 'Failed to fetch Tgju SEK: ' . $response->get_error_message());
        }
        
        $body = wp_remote_retrieve_body($response);
        if (empty($body)) {
            return new \WP_Error('empty_response', 'Empty response from Tgju SEK');
        }
        
        $rate = $this->extract_rate_from_html($body);
        
        if (!$rate) {
            return new \WP_Error('extraction_failed', 'Could not extract valid SEK rate from Tgju');
        }
        
        if (!$this->validate_rate($rate)) {
            return new \WP_Error('validation_failed', 'Extracted rate failed validation: ' . $rate);
        }
        
        return $this->format_rate_data($rate, current_time('mysql'));
    }
    
    private function extract_rate_from_html($html) {
        $dom = new \DOMDocument();
        
        libxml_use_internal_errors(true);
        $dom->loadHTML($html);
        libxml_clear_errors();
        
        $xpath = new \DOMXPath($dom);
        
        $candidates = [];
        
        $candidates = array_merge($candidates, $this->extract_by_selectors($xpath));
        $candidates = array_merge($candidates, $this->extract_by_patterns($html));
        $candidates = array_merge($candidates, $this->extract_from_json($html));
        
        return $this->select_best_candidate($candidates);
    }
    
    private function extract_by_selectors($xpath) {
        $candidates = [];
        
        $selectors = [
            // Target the EXACT table structure from the real website - HIGHEST PRIORITY
            '//td[@class="text-right" and contains(text(), "نرخ فعلی")]/following-sibling::td[@class="text-left"]/text()',
            '//td[contains(text(), "نرخ فعلی")]/following-sibling::td[@class="text-left"]/text()',
            '//td[@class="text-right" and text()="نرخ فعلی"]/following-sibling::td[@class="text-left"]/text()',
            
            // Also target the span variant
            '//td[@class="text-right" and contains(text(), "نرخ فعلی")]/following-sibling::td[@class="text-left"]/span/text()',
            '//td[contains(text(), "نرخ فعلی")]/following-sibling::td[@class="text-left"]/span/text()',
            
            // More general table selectors
            '//tr[td[contains(text(), "نرخ فعلی")]]/td[2]/text()',
            '//tr[td[text()="نرخ فعلی"]]/td[2]/text()',
            '//tr[td[contains(text(), "نرخ فعلی")]]/td[2]/span/text()',
            '//tr[td[text()="نرخ فعلی"]]/td[2]/span/text()',
            
            // Fallback selectors
            '//td[contains(text(), "نرخ فعلی")]/following-sibling::td[1]/text()',
            '//td[text()="نرخ فعلی"]/following-sibling::td[1]/text()',
            '//td[contains(text(), "نرخ فعلی")]/following-sibling::td[1]/span/text()',
            '//td[text()="نرخ فعلی"]/following-sibling::td[1]/span/text()',
            
            '//span[contains(@class, "value") and contains(@class, "price")]/text()',
            '//div[contains(@class, "market") and contains(@class, "price")]/text()',
            '//span[contains(@class, "price") and not(contains(@class, "change"))]/text()',
            '//div[contains(@class, "info-row")]/span[contains(@class, "value")]/text()',
            '//td[contains(text(), "قیمت")]/following-sibling::td/text()',
            '//div[contains(@class, "sek") and contains(@class, "price")]/text()',
            '//span[contains(@class, "sek")]/text()',
            '//div[contains(@class, "profile") and contains(@class, "price")]/text()',
            '//strong[contains(text(), "کرون")]/following-sibling::text()',
            '//div[contains(@class, "data") and contains(@class, "value")]/text()',
        ];
        
        foreach ($selectors as $selector) {
            try {
                $nodes = $xpath->query($selector);
                foreach ($nodes as $node) {
                    $text = trim($node->nodeValue);
                    $rate = $this->clean_and_parse_rate($text);
                    if ($rate) {
                        $candidates[] = [
                            'rate' => $rate,
                            'source' => 'selector: ' . $selector,
                            'text' => $text
                        ];
                    }
                }
            } catch (Exception $e) {
                continue;
            }
        }
        
        return $candidates;
    }
    
    private function extract_by_patterns($html) {
        $candidates = [];
        
        $patterns = [
            // Target the EXACT table structure from real website - HIGHEST PRIORITY
            '/نرخ فعلی<\/td>\s*<td[^>]*>([0-9,]+)<\/td>/i',
            '/نرخ فعلی<\/td>\s*<td[^>]*><span[^>]*>([0-9,]+)<\/span><\/td>/i',
            '/<td[^>]*>نرخ فعلی<\/td>\s*<td[^>]*>([0-9,]+)<\/td>/i',
            '/<td[^>]*>نرخ فعلی<\/td>\s*<td[^>]*><span[^>]*>([0-9,]+)<\/span><\/td>/i',
            '/نرخ فعلی.*?([0-9,]+)\s*<\/td>/i',
            
            // More specific patterns for the table with class attributes
            '/<td[^>]*class="text-right"[^>]*>نرخ فعلی<\/td>\s*<td[^>]*class="text-left"[^>]*>([0-9,]+)<\/td>/i',
            '/<td[^>]*class="text-right"[^>]*>نرخ فعلی<\/td>\s*<td[^>]*class="text-left"[^>]*><span[^>]*>([0-9,]+)<\/span><\/td>/i',
            
            // More specific patterns for the table
            '/<tr[^>]*>.*?نرخ فعلی.*?<\/td>\s*<td[^>]*>([0-9,]+)<\/td>/is',
            '/<tr[^>]*>.*?نرخ فعلی.*?<\/td>\s*<td[^>]*><span[^>]*>([0-9,]+)<\/span><\/td>/is',
            
            // JSON patterns
            '/"price_sek"\s*:\s*"([0-9,]+)"/i',
            '/"sek"\s*:\s*"([0-9,]+)"/i',
            '/"price"\s*:\s*"([0-9,]+)"/i',
            '/"value"\s*:\s*"([0-9,]+)"/i',
            '/data-price="([0-9,]+)"/i',
            '/data-value="([0-9,]+)"/i',
            
            // Text patterns
            '/قیمت\s*کرون[:\s]*([0-9,]+)/i',
            '/کرون[:\s]*([0-9,]+)/i',
            '/(\d{1,3}[,\s]?\d{3}[,\s]?\d{3})\s*تومان/i',
            '/(\d{1,3}[,\s]?\d{3}[,\s]?\d{3})\s*ریال/i',
        ];
        
        foreach ($patterns as $pattern) {
            if (preg_match_all($pattern, $html, $matches)) {
                foreach ($matches[1] as $match) {
                    $rate = $this->clean_and_parse_rate($match);
                    if ($rate) {
                        $candidates[] = [
                            'rate' => $rate,
                            'source' => 'pattern: ' . $pattern,
                            'text' => $match
                        ];
                    }
                }
            }
        }
        
        return $candidates;
    }
    
    private function extract_from_json($html) {
        $candidates = [];
        
        $json_patterns = [
            '/window\.__INITIAL_STATE__\s*=\s*({.+?});/s',
            '/window\.data\s*=\s*({.+?});/s',
            '/var\s+data\s*=\s*({.+?});/s',
            '/"data":\s*({.+?})/s',
        ];
        
        foreach ($json_patterns as $pattern) {
            if (preg_match($pattern, $html, $matches)) {
                try {
                    $json_data = json_decode($matches[1], true);
                    if ($json_data) {
                        $rate = $this->extract_rate_from_json_data($json_data);
                        if ($rate) {
                            $candidates[] = [
                                'rate' => $rate,
                                'source' => 'json: ' . $pattern,
                                'text' => json_encode($rate)
                            ];
                        }
                    }
                } catch (Exception $e) {
                    continue;
                }
            }
        }
        
        return $candidates;
    }
    
    private function select_best_candidate($candidates) {
        if (empty($candidates)) {
            return false;
        }
        
        usort($candidates, function($a, $b) {
            $score_a = $this->calculate_candidate_score($a);
            $score_b = $this->calculate_candidate_score($b);
            return $score_b - $score_a;
        });
        
        $best = $candidates[0];
        
        if ($this->calculate_candidate_score($best) < 3) {
            return false;
        }
        
        error_log('Tgju: Selected rate ' . $best['rate'] . ' from ' . $best['source']);
        
        return $best['rate'];
    }
    
    private function clean_and_parse_rate($text) {
        if (empty($text)) {
            return false;
        }
        
        $text = preg_replace('/[^\d,]/', '', $text);
        $text = str_replace(',', '', $text);
        
        if (!is_numeric($text)) {
            return false;
        }
        
        $rate = (int) $text;
        
        // Allow broader range for current market rates
        if ($rate < 30000 || $rate > 3000000) {
            return false;
        }
        
        return $rate;
    }
    
    private function extract_rate_from_json_data($data) {
        if (!is_array($data)) {
            return false;
        }
        
        $keys_to_check = [
            'price_sek',
            'sek',
            'krona',
            'price',
            'value',
            'rate'
        ];
        
        foreach ($keys_to_check as $key) {
            if (isset($data[$key])) {
                $rate = $this->clean_and_parse_rate($data[$key]);
                if ($rate) {
                    return $rate;
                }
            }
        }
        
        $this->search_nested_array($data, $keys_to_check, $found_rates);
        
        if (!empty($found_rates)) {
            return $found_rates[0];
        }
        
        return false;
    }
    
    private function search_nested_array($array, $keys, &$found_rates) {
        if (!is_array($array)) {
            return;
        }
        
        foreach ($array as $key => $value) {
            if (in_array($key, $keys) && is_string($value)) {
                $rate = $this->clean_and_parse_rate($value);
                if ($rate) {
                    $found_rates[] = $rate;
                }
            } elseif (is_array($value)) {
                $this->search_nested_array($value, $keys, $found_rates);
            }
        }
    }
    
    private function calculate_candidate_score($candidate) {
        $score = 0;
        $rate = $candidate['rate'];
        $source = $candidate['source'];
        $text = $candidate['text'];
        
        // Prioritize realistic SEK rates (120,000 - 150,000 IRR range)
        if ($rate >= 120000 && $rate <= 150000) {
            $score += 8;
        } elseif ($rate >= 100000 && $rate <= 200000) {
            $score += 5;
        } elseif ($rate >= 30000 && $rate <= 600000) {
            $score += 3;
        } elseif ($rate >= 20000 && $rate <= 1000000) {
            $score += 2;
        }
        
        // VERY HIGH priority for "نرخ فعلی" (current rate) pattern
        if (strpos($source, 'نرخ فعلی') !== false) {
            $score += 15;
        }
        
        // VERY HIGH priority for table-based selectors with "نرخ فعلی"
        if (strpos($source, 'td[contains(text(), "نرخ فعلی")]') !== false || 
            strpos($source, 'td[text()="نرخ فعلی"]') !== false) {
            $score += 12;
        }
        
        // High priority for patterns targeting "نرخ فعلی"
        if (strpos($source, 'pattern: نرخ فعلی') !== false) {
            $score += 10;
        }
        
        if (strpos($source, 'json') !== false) {
            $score += 4;
        }
        
        if (strpos($source, 'selector') !== false) {
            $score += 3;
        }
        
        if (strpos($source, 'sek') !== false || strpos($source, 'price_sek') !== false) {
            $score += 3;
        }
        
        if (strpos($source, 'krona') !== false || strpos($source, 'کرون') !== false) {
            $score += 2;
        }
        
        if (strpos($text, 'تومان') !== false || strpos($text, 'ریال') !== false) {
            $score += 2;
        }
        
        // Bonus for longer text (likely more specific)
        if (strlen($text) > 5) {
            $score += 1;
        }
        
        // Bonus for comma-separated numbers (proper formatting)
        if (strpos($text, ',') !== false) {
            $score += 1;
        }
        
        return $score;
    }
    
    public function validate_rate($rate) {
        if (!is_numeric($rate)) {
            return false;
        }
        
        $rate = (float) $rate;
        
        // Updated validation for realistic SEK rates in Iran
        // Allow broader range: 50,000 - 3,000,000 IRR
        // This accommodates both older formats and current market rates
        if ($rate < 50000 || $rate > 3000000) {
            return false;
        }
        
        return true;
    }
    
    public function format_rate_data($rate, $timestamp) {
        return [
            'source' => $this->get_source_name(),
            'currency' => $this->get_currency_code(),
            'rate' => $rate,
            'timestamp' => $timestamp,
            'url' => $this->get_source_url()
        ];
    }
}