<?php
namespace AliKhaleghi\CurrenciesInIran\Sources\USD;

use AliKhaleghi\CurrenciesInIran\Sources\CurrencyInterface;

if (!defined('ABSPATH')) {
    exit;
}

class EghtesadNews implements CurrencyInterface {
    
    private $url = 'https://www.eghtesadnews.com/markets/dollar';
    private $user_agent;
    
    public function __construct() {
        $this->user_agent = 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36';
    }
    
    public function get_currency_code() {
        return 'USD';
    }
    
    public function get_source_name() {
        return 'EghtesadNews';
    }
    
    public function get_source_url() {
        return $this->url;
    }
    
    public function get_user_agent() {
        return $this->user_agent;
    }
    
    public function get_request_headers() {
        return [
            'Accept' => 'text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
            'Accept-Language' => 'fa-IR,fa;q=0.8,en-US;q=0.5,en;q=0.3',
            'Accept-Encoding' => 'gzip, deflate',
            'Connection' => 'keep-alive',
            'Upgrade-Insecure-Requests' => '1',
            'Cache-Control' => 'max-age=0',
        ];
    }
    
    public function get_timeout() {
        return 15;
    }
    
    public function scrape_rate() {
        $response = wp_remote_get($this->url, [
            'timeout' => $this->get_timeout(),
            'user-agent' => $this->get_user_agent(),
            'headers' => $this->get_request_headers(),
        ]);
        
        if (is_wp_error($response)) {
            return new \WP_Error('http_error', 'Failed to fetch EghtesadNews: ' . $response->get_error_message());
        }
        
        $body = wp_remote_retrieve_body($response);
        if (empty($body)) {
            return new \WP_Error('empty_response', 'Empty response from EghtesadNews');
        }
        
        $rate = $this->extract_rate_from_html($body);
        
        if (!$rate) {
            return new \WP_Error('extraction_failed', 'Could not extract valid USD rate from EghtesadNews');
        }
        
        if (!$this->validate_rate($rate)) {
            return new \WP_Error('validation_failed', 'Extracted rate failed validation: ' . $rate);
        }
        
        return $this->format_rate_data($rate, current_time('mysql'));
    }
    
    private function extract_rate_from_html($html) {
        $dom = new \DOMDocument();
        
        libxml_use_internal_errors(true);
        $dom->loadHTML($html);
        libxml_clear_errors();
        
        $xpath = new \DOMXPath($dom);
        
        $candidates = [];
        
        $candidates = array_merge($candidates, $this->extract_by_selectors($xpath));
        $candidates = array_merge($candidates, $this->extract_by_patterns($html));
        
        return $this->select_best_candidate($candidates);
    }
    
    private function extract_by_selectors($xpath) {
        $candidates = [];
        
        $selectors = [
            // Target the EXACT table structure from EghtesadNews - HIGHEST PRIORITY
            '//table[contains(@class, "price-tables")]//tr[td[contains(text(), "دلار") and not(contains(text(), "کانادا")) and not(contains(text(), "صرافی")) and not(contains(text(), "متشکل")) and not(contains(text(), "سنا"))]]/td[@class="textcenter"]/text()',
            '//table[contains(@class, "price-tables")]//tr[td[text()="دلار"]]/td[@class="textcenter"]/text()',
            '//table[contains(@class, "price-tables")]//tr[td[contains(text(), "دلار") and not(contains(text(), "کانادا"))]]/td[2]/text()',
            '//table[contains(@class, "price-tables")]//tr[td[text()="دلار"]]/td[2]/text()',
            
            // More specific table selectors for the main dollar row
            '//tr[td[contains(text(), "دلار") and not(contains(text(), "کانادا")) and not(contains(text(), "صرافی")) and not(contains(text(), "متشکل")) and not(contains(text(), "سنا"))]]/td[@class="textcenter"]/text()',
            '//tr[td[text()="دلار"]]/td[@class="textcenter"]/text()',
            '//tr[td[contains(text(), "دلار") and not(contains(text(), "کانادا"))]]/td[2]/text()',
            '//tr[td[text()="دلار"]]/td[2]/text()',
            
            // Fallback selectors
            '//div[contains(@class, "dollar") and contains(@class, "price")]/text()',
            '//span[contains(@class, "dollar") and contains(@class, "price")]/text()',
            '//div[contains(@class, "market") and contains(@class, "data")]//span[contains(text(), "دلار")]/following-sibling::span/text()',
            '//td[contains(text(), "دلار")]/following-sibling::td/text()',
            '//div[contains(@class, "info")]/span[contains(text(), "قیمت")]/following-sibling::span/text()',
            '//div[contains(@class, "currency") and contains(@class, "dollar")]/span[contains(@class, "value")]/text()',
            '//span[contains(@class, "price") and contains(text(), "تومان")]/text()',
            '//div[contains(@class, "rate")]/text()',
            '//strong[contains(text(), "دلار")]/following-sibling::text()',
            '//b[contains(text(), "دلار")]/following-sibling::text()',
            
            // Additional selectors for better coverage
            '//div[contains(@class, "price")]/span[contains(text(), "دلار")]/text()',
            '//span[contains(@class, "price") and contains(@class, "dollar")]/text()',
            '//div[contains(@class, "market")]/span[contains(@class, "price")]/text()',
            '//table//td[contains(text(), "دلار")]/following-sibling::td[1]/text()',
            '//div[contains(@class, "dollar-price")]/text()',
            '//span[contains(@class, "usd") and contains(@class, "price")]/text()',
        ];
        
        foreach ($selectors as $selector) {
            try {
                $nodes = $xpath->query($selector);
                foreach ($nodes as $node) {
                    $text = trim($node->nodeValue);
                    $rate = $this->clean_and_parse_rate($text);
                    if ($rate) {
                        $candidates[] = [
                            'rate' => $rate,
                            'source' => 'selector: ' . $selector,
                            'text' => $text
                        ];
                    }
                }
            } catch (Exception $e) {
                continue;
            }
        }
        
        return $candidates;
    }
    
    private function extract_by_patterns($html) {
        $candidates = [];
        
        $patterns = [
            // Target the EXACT table structure from EghtesadNews - HIGHEST PRIORITY
            '/<tr>\s*<td><i[^>]*><\/i>\s*دلار<\/td>\s*<td[^>]*class="textcenter"[^>]*>([0-9,]+)\s*<span[^>]*>تومان<\/span><\/td>/i',
            '/<tr>\s*<td>دلار<\/td>\s*<td[^>]*class="textcenter"[^>]*>([0-9,]+)\s*<span[^>]*>تومان<\/span><\/td>/i',
            '/دلار<\/td>\s*<td[^>]*class="textcenter"[^>]*>([0-9,]+)\s*<span[^>]*>تومان<\/span><\/td>/i',
            '/<td[^>]*class="textcenter"[^>]*>([0-9,]+)\s*<span[^>]*>تومان<\/span><\/td>/i',
            
            // More specific patterns for the main dollar row
            '/<tr>\s*<td>دلار<\/td>\s*<td[^>]*>([0-9,]+)\s*<span[^>]*>تومان<\/span><\/td>/i',
            '/دلار<\/td>\s*<td[^>]*>([0-9,]+)\s*<span[^>]*>تومان<\/span><\/td>/i',
            
            // Fallback patterns
            '/قیمت\s*دلار[:\s]*([0-9,]+)/i',
            '/دلار[:\s]*([0-9,]+)/i',
            '/dollar[:\s]*([0-9,]+)/i',
            '/(\d{1,3}[,\s]?\d{3}[,\s]?\d{3})\s*تومان/i',
            '/(\d{1,3}[,\s]?\d{3}[,\s]?\d{3})\s*ریال/i',
            '/"price_dollar_rl"\s*:\s*"([0-9,]+)"/i',
            '/"dollar"\s*:\s*"([0-9,]+)"/i',
            '/data-price="([0-9,]+)"/i',
            '/value="([0-9,]+)"/i',
            
            // Additional patterns
            '/"dollar_price"\s*:\s*"([0-9,]+)"/i',
            '/"usd_price"\s*:\s*"([0-9,]+)"/i',
            '/price.*?dollar.*?([0-9,]+)/i',
            '/dollar.*?price.*?([0-9,]+)/i',
            '/قیمت.*?([0-9,]+).*?دلار/i',
            '/([0-9,]+).*?قیمت.*?دلار/i',
        ];
        
        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $html, $matches)) {
                $rate = $this->clean_and_parse_rate($matches[1]);
                if ($rate) {
                    $candidates[] = [
                        'rate' => $rate,
                        'source' => 'pattern: ' . $pattern,
                        'text' => $matches[1]
                    ];
                }
            }
        }
        
        return $candidates;
    }
    
    private function clean_and_parse_rate($text) {
        if (empty($text)) {
            return false;
        }
        
        $text = preg_replace('/[^\d,]/', '', $text);
        $text = str_replace(',', '', $text);
        
        if (!is_numeric($text)) {
            return false;
        }
        
        $rate = (int) $text;
        
        // Convert Toman to Rial if needed (1 Toman = 10 Rial)
        // EghtesadNews shows prices in Toman, we need Rial
        $rate = $rate * 10;
        
        // Allow broader range for current market rates
        if ($rate < 30000 || $rate > 3000000) {
            return false;
        }
        
        return $rate;
    }
    
    private function select_best_candidate($candidates) {
        if (empty($candidates)) {
            return false;
        }
        
        usort($candidates, function($a, $b) {
            $score_a = $this->calculate_candidate_score($a);
            $score_b = $this->calculate_candidate_score($b);
            return $score_b - $score_a;
        });
        
        $best = $candidates[0];
        
        if ($this->calculate_candidate_score($best) < 3) {
            return false;
        }
        
        error_log('EghtesadNews: Selected rate ' . $best['rate'] . ' from ' . $best['source']);
        
        return $best['rate'];
    }
    
    private function calculate_candidate_score($candidate) {
        $score = 0;
        $rate = $candidate['rate'];
        $source = $candidate['source'];
        $text = $candidate['text'];
        
        // Prioritize realistic USD rates (1,200,000 - 1,300,000 IRR range)
        if ($rate >= 1200000 && $rate <= 1300000) {
            $score += 8;
        } elseif ($rate >= 1000000 && $rate <= 1500000) {
            $score += 5;
        } elseif ($rate >= 50000 && $rate <= 3000000) {
            $score += 3;
        }
        
        // VERY HIGH priority for exact table structure patterns
        if (strpos($source, 'price-tables') !== false) {
            $score += 15;
        }
        
        // HIGH priority for main dollar row (excluding other dollar types)
        if (strpos($source, 'not(contains(text(), "کانادا"))') !== false || 
            strpos($source, 'not(contains(text(), "صرافی"))') !== false) {
            $score += 12;
        }
        
        // HIGH priority for patterns targeting the exact structure
        if (strpos($source, 'pattern: <tr>') !== false || 
            strpos($source, 'دلار<\/td>') !== false) {
            $score += 10;
        }
        
        if (strpos($source, 'selector') !== false) {
            $score += 3;
        }
        
        if (strpos($source, 'dollar') !== false || strpos($source, 'دلار') !== false) {
            $score += 2;
        }
        
        if (strpos($text, 'تومان') !== false || strpos($text, 'ریال') !== false) {
            $score += 2;
        }
        
        if (strlen($text) > 5) {
            $score += 1;
        }
        
        // Bonus for comma-separated numbers (proper formatting)
        if (strpos($text, ',') !== false) {
            $score += 1;
        }
        
        // Bonus for "تومان" unit (correct currency unit)
        if (strpos($text, 'تومان') !== false) {
            $score += 2;
        }
        
        return $score;
    }
    
    public function validate_rate($rate) {
        if (!is_numeric($rate)) {
            return false;
        }
        
        $rate = (float) $rate;
        
        // Updated validation for realistic USD rates in Iran
        // Allow broader range: 50,000 - 3,000,000 IRR
        // This accommodates both older formats and current market rates
        if ($rate < 50000 || $rate > 3000000) {
            return false;
        }
        
        return true;
    }
    
    public function format_rate_data($rate, $timestamp) {
        return [
            'source' => $this->get_source_name(),
            'currency' => $this->get_currency_code(),
            'rate' => $rate,
            'timestamp' => $timestamp,
            'url' => $this->get_source_url()
        ];
    }
}