# **📋 ARCHITECTURE DOCUMENT**

# **Remote Media Storage**

---

# **1. Overview**

Remote Media Storage is a WordPress plugin that provides an abstraction layer for storing media files on external storage systems while maintaining seamless integration with WordPress's native media management.

---

# **2. Core Architecture**

## **2.1 Plugin Structure**

```
/remote-media-storage
    /includes
        /drivers
            StorageDriverInterface.php
            DriverFTP.php
            DriverSFTP.php
            DriverFTPS.php
        /storage
            StorageSpace.php
            StorageManager.php
            StorageValidator.php
        /managers
            FilePipeline.php
            MigrationManager.php
            UrlManager.php
            BackupManager.php
            QueueManager.php
        /libraries
            Logger.php
            Cache.php
            Security.php
            Utilities.php
        /integration
            MediaLibraryIntegration.php
            WordPressHooks.php
        /admin
            AdminInterface.php
            SettingsPage.php
            Dashboard.php
    /assets
        /css
            admin.css
            frontend.css
        /js
            admin.js
            upload.js
            migration.js
    /templates
        /admin
            settings.php
            storage-spaces.php
            migration.php
        /public
            media-display.php
    /languages
        remote-media-storage.pot
    remote-media-storage.php
    uninstall.php
```

## **2.2 Key Components**

### **Storage Driver Interface**
- Abstracts storage operations
- Supports FTP, SFTP, FTPS protocols
- Extensible for future drivers (S3, WebDAV, etc.)

### **File Pipeline**
- Queue-based processing
- Handles uploads, migrations, deletions
- Provides retry mechanisms and error handling

### **URL Manager**
- Rewrites media URLs to point to remote storage
- Handles URL updates and rebuilds
- Maintains WordPress compatibility

### **Migration Manager**
- Transfers existing media to remote storage
- Batch processing with progress tracking
- Preserves metadata and relationships

---

# **3. Database Schema**

## **3.1 Storage Spaces Table**

```sql
CREATE TABLE rsm_storage_spaces (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    type ENUM('ftp', 'sftp', 'ftps') NOT NULL,
    host VARCHAR(255) NOT NULL,
    port INT NOT NULL DEFAULT 21,
    username VARCHAR(255) NOT NULL,
    password TEXT NOT NULL,
    root_path VARCHAR(255) DEFAULT '/',
    public_url VARCHAR(500) NOT NULL,
    status ENUM('active', 'inactive', 'error') DEFAULT 'active',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);
```

## **3.2 File Index Table**

```sql
CREATE TABLE rsm_file_index (
    id INT AUTO_INCREMENT PRIMARY KEY,
    wp_attachment_id BIGINT NOT NULL UNIQUE,
    storage_space_id INT NOT NULL,
    relative_path VARCHAR(500) NOT NULL,
    full_url VARCHAR(500) NOT NULL,
    file_size BIGINT,
    checksum VARCHAR(64),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (storage_space_id) REFERENCES rsm_storage_spaces(id)
);
```

## **3.3 Queue Table**

```sql
CREATE TABLE rsm_queue (
    id INT AUTO_INCREMENT PRIMARY KEY,
    action ENUM('upload', 'delete', 'replace', 'migrate') NOT NULL,
    storage_space_id INT NOT NULL,
    wp_attachment_id BIGINT,
    status ENUM('pending', 'processing', 'done', 'error') DEFAULT 'pending',
    error_message TEXT,
    retry_count INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT INTEGER,
    processed_at TIMESTAMP NULL,
    FOREIGN KEY (storage_space_id) REFERENCES rsm_storage_spaces(id)
);
```

## **3.4 Logs Table (Pro)**

```sql
CREATE TABLE rsm_logs (
    id INT AUTO_INCREMENT PRIMARY KEY,
    action VARCHAR(50) NOT NULL,
    storage_space_id INT,
    wp_attachment_id BIGINT,
    user_id BIGINT,
    details TEXT,
    status ENUM('success', 'error') NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);
```

---

# **4. Class Architecture**

## **4.1 Namespace Structure**

All classes use the base namespace: `\AliKhaleghi\RemoteMediaStorage`

```
\AliKhaleghi\RemoteMediaStorage\
├── Drivers\
│   ├── StorageDriverInterface
│   ├── FTP
│   ├── SFTP
│   └── FTPS
├── Storage\
│   ├── StorageSpace
│   ├── StorageManager
│   └── StorageValidator
├── Managers\
│   ├── FilePipeline
│   ├── MigrationManager
│   ├── UrlManager
│   ├── BackupManager
│   └── QueueManager
├── Libraries\
│   ├── Logger
│   ├── Cache
│   ├── Security
│   └── Utilities
├── Integration\
│   ├── MediaLibraryIntegration
│   └── WordPressHooks
└── Admin\
    ├── AdminInterface
    ├── SettingsPage
    └── Dashboard
```

## **4.2 Storage Driver Interface**

```php
<?php
// includes/drivers/StorageDriverInterface.php
namespace AliKhaleghi\RemoteMediaStorage\Drivers;

interface StorageDriverInterface {
    public function connect(): bool;
    public function disconnect(): void;
    public function testConnection(): array;
    public function uploadFile(string $localPath, string $remotePath): bool;
    public function deleteFile(string $remotePath): bool;
    public function fileExists(string $remotePath): bool;
    public function getFileSize(string $remotePath): int;
    public function createDirectory(string $path): bool;
    public function listFiles(string $directory): array;
    public function getPermissions(string $path): string;
}
```

## **4.3 Storage Manager**

```php
<?php
// includes/storage/StorageManager.php
namespace AliKhaleghi\RemoteMediaStorage\Storage;

use AliKhaleghi\RemoteMediaStorage\Drivers\StorageDriverInterface;
use AliKhaleghi\RemoteMediaStorage\Storage\StorageSpace;

class StorageManager {
    private $drivers = [];
    private $activeStorage;
    
    public function __construct();
    public function registerDriver(string $type, StorageDriverInterface $driver): void;
    public function createStorage(array $config): StorageSpace;
    public function getStorage(int $id): ?StorageSpace;
    public function getActiveStorage(): ?StorageSpace;
    public function testStorage(int $id): array;
}
```

## **4.4 Storage Space**

```php
<?php
// includes/storage/StorageSpace.php
namespace AliKhaleghi\RemoteMediaStorage\Storage;

use AliKhaleghi\RemoteMediaStorage\Drivers\StorageDriverInterface;

class StorageSpace {
    private $id;
    private $config;
    private $driver;
    private $status;
    
    public function __construct(int $id, array $config);
    public function getId(): int;
    public function getName(): string;
    public function getType(): string;
    public function getStatus(): string;
    public function testConnection(): array;
    public function getPublicUrl(): string;
    public function getRemotePath(string $relativePath): string;
    public function isHealthy(): bool;
}
```

## **4.5 File Pipeline**

```php
<?php
// includes/managers/FilePipeline.php
namespace AliKhaleghi\RemoteMediaStorage\Managers;

use AliKhaleghi\RemoteMediaStorage\Managers\QueueManager;
use AliKhaleghi\RemoteMediaStorage\Storage\StorageManager;

class FilePipeline {
    private $queueManager;
    private $storageManager;
    
    public function __construct(QueueManager $queue, StorageManager $storage);
    public function addJob(string $action, int $attachmentId, int $storageId): int;
    public function processQueue(): void;
    public function getJobStatus(int $jobId): array;
    public function retryFailedJobs(): void;
    public function cancelJob(int $jobId): bool;
}
```

## **4.6 Queue Manager**

```php
<?php
// includes/managers/QueueManager.php
namespace AliKhaleghi\RemoteMediaStorage\Managers;

class QueueManager {
    private $maxRetries = 3;
    private $batchSize = 10;
    
    public function addJob(array $jobData): int;
    public function getNextJob(): ?array;
    public function updateJobStatus(int $jobId, string $status, array $data = []): bool;
    public function getJobsByStatus(string $status): array;
    public function getJobStats(): array;
    public function cleanupCompletedJobs(): int;
}
```

## **4.7 Driver Implementations**

```php
<?php
// includes/drivers/FTP.php
namespace AliKhaleghi\RemoteMediaStorage\Drivers;

use AliKhaleghi\RemoteMediaStorage\Drivers\StorageDriverInterface;

class FTP implements StorageDriverInterface {
    private $connection;
    private $config;
    
    public function __construct(array $config);
    public function connect(): bool;
    public function disconnect(): void;
    public function testConnection(): array;
    public function uploadFile(string $localPath, string $remotePath): bool;
    public function deleteFile(string $remotePath): bool;
    public function fileExists(string $remotePath): bool;
    public function getFileSize(string $remotePath): int;
    public function createDirectory(string $path): bool;
    public function listFiles(string $directory): array;
    public function getPermissions(string $path): string;
}
```

## **4.8 Integration Classes**

```php
<?php
// includes/integration/MediaLibraryIntegration.php
namespace AliKhaleghi\RemoteMediaStorage\Integration;

use AliKhaleghi\RemoteMediaStorage\Storage\StorageManager;
use AliKhaleghi\RemoteMediaStorage\Managers\FilePipeline;

class MediaLibraryIntegration {
    private $storageManager;
    private $filePipeline;
    
    public function __construct(StorageManager $storage, FilePipeline $pipeline);
    public function interceptUpload(array $upload): array;
    public function rewriteAttachmentUrl(string $url, int $attachmentId): string;
    public function handleAttachmentDeletion(int $attachmentId): void;
}
```

---

# **5. Integration Points**

## **5.1 WordPress Hooks Integration**

```php
<?php
// includes/integration/WordPressHooks.php
namespace AliKhaleghi\RemoteMediaStorage\Integration;

use AliKhaleghi\RemoteMediaStorage\Integration\MediaLibraryIntegration;

class WordPressHooks {
    private $mediaIntegration;
    
    public function __construct(MediaLibraryIntegration $integration);
    public function registerHooks(): void;
    
    // Hook implementations
    public function handleUpload(array $upload): array;                    // wp_handle_upload
    public function generateMetadata(array $metadata, int $attachmentId): array; // wp_generate_attachment_metadata
    public function deleteAttachment(int $attachmentId): void;             // wp_delete_attachment
    public function rewriteAttachmentUrl(string $url, int $attachmentId): string; // wp_get_attachment_url
    public function addAdminMenu(): void;                                   // admin_menu
}
```

## **5.2 Hook Registration**

```php
<?php
// remote-media-storage.php (main plugin file)
namespace AliKhaleghi\RemoteMediaStorage;

use AliKhaleghi\RemoteMediaStorage\Integration\WordPressHooks;

function initializePlugin(): void {
    $hooks = new WordPressHooks(/* dependencies */);
    $hooks->registerHooks();
}

// Register hooks
add_action('plugins_loaded', '\AliKhaleghi\RemoteMediaStorage\initializePlugin');
```

---

# **6. Security Considerations**

## **6.1 Credential Storage**
- Encrypted password storage
- WordPress options API
- Key derivation for encryption

## **6.2 File Validation**
- MIME type checking
- File size limits
- Path traversal prevention

## **6.3 Access Control**
- User capability checks
- Storage space isolation
- Multi-site support

---

# **7. Performance Optimization**

## **7.1 Caching**
- URL caching
- Connection pooling
- Thumbnail caching (Pro)

## **7.2 Queue Processing**
- Background processing
- Batch operations
- Rate limiting

## **7.3 Error Handling**
- Retry mechanisms
- Fallback strategies
- Comprehensive logging

---

# **8. Extensibility**

## **8.1 Driver System**

```php
<?php
// includes/storage/StorageValidator.php
namespace AliKhaleghi\RemoteMediaStorage\Storage;

use AliKhaleghi\RemoteMediaStorage\Drivers\StorageDriverInterface;

class StorageValidator {
    public static function validateConfig(array $config): array;
    public static function validateDriver(StorageDriverInterface $driver): bool;
    public static function sanitizePath(string $path): string;
    public static function validateUrl(string $url): bool;
}
```

## **8.2 Hook System**

```php
<?php
// includes/libraries/Utilities.php
namespace AliKhaleghi\RemoteMediaStorage\Libraries;

class Utilities {
    public static function registerHook(string $hook, callable $callback, int $priority = 10): void;
    public static function applyFilters(string $hook, $value, ...$args);
    public static function doAction(string $hook, ...$args): void;
    public static function getHookPriority(string $hook, callable $callback): int;
}
```

## **8.3 Autoloader**

```php
<?php
// includes/libraries/Autoloader.php
namespace AliKhaleghi\RemoteMediaStorage\Libraries;

class Autoloader {
    private static $namespace = 'AliKhaleghi\\RemoteMediaStorage\\';
    private static $baseDir;
    
    public static function register(string $baseDir): void;
    public static function load(string $className): bool;
    public static function getClassPath(string $className): string;
}
```

---

# **9. Deployment**

## **9.1 Requirements**
- WordPress 5.0+
- PHP 7.4+
- FTP/SFTP extensions
- MySQL 5.7+

## **9.2 Installation**
- Standard plugin installation
- Database migration
- Configuration wizard

---

# **10. Monitoring & Maintenance**

## **10.1 Health Checks**
- Storage connectivity
- Queue status
- Error rates

## **10.2 Maintenance Tasks**
- Log rotation
- Queue cleanup
- Cache refresh