# **🗄️ DATABASE SCHEMA**

# **Remote Media Storage**

---

# **1. Overview**

Remote Media Storage uses custom database tables to track storage configurations, file locations, queue operations, and audit logs. This document defines the complete database schema.

---

# **2. Table Overview**

- `rsm_storage_spaces` - Storage space configurations
- `rsm_file_index` - File tracking and metadata
- `rsm_queue` - Operation queue management
- `rsm_logs` - Audit and activity logs
- `rsm_settings` - Plugin configuration

---

# **3. Storage Spaces Table**

## **3.1 rsm_storage_spaces**

```sql
CREATE TABLE rsm_storage_spaces (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    type ENUM('ftp', 'sftp', 'ftps') NOT NULL,
    host VARCHAR(255) NOT NULL,
    port INT NOT NULL DEFAULT 21,
    username VARCHAR(255) NOT NULL,
    password TEXT NOT NULL,
    root_path VARCHAR(255) DEFAULT '/',
    public_url VARCHAR(500) NOT NULL,
    status ENUM('active', 'inactive', 'error', 'testing') DEFAULT 'testing',
    timeout INT DEFAULT 30,
    passive_mode BOOLEAN DEFAULT TRUE,
    ssl_verify BOOLEAN DEFAULT TRUE,
    max_connections INT DEFAULT 3,
    retry_count INT DEFAULT 3,
    chunk_size INT DEFAULT 8192,
    last_test TIMESTAMP NULL,
    last_error TEXT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    INDEX idx_status (status),
    INDEX idx_type (type),
    INDEX idx_host (host)
);
```

### **Field Descriptions**

- **id**: Unique identifier
- **name**: Human-readable name
- **type**: Storage protocol
- **host**: Server hostname/IP
- **port**: Connection port
- **username**: Login username
- **password**: Encrypted password
- **root_path**: Base directory
- **public_url**: Base URL for files
- **status**: Current operational status
- **timeout**: Connection timeout
- **passive_mode**: FTP passive mode
- **ssl_verify**: SSL certificate verification
- **max_connections**: Concurrent connections
- **retry_count**: Maximum retry attempts
- **chunk_size**: Transfer chunk size
- **last_test**: Last connectivity test
- **last_error**: Last error message

---

# **4. File Index Table**

## **4.1 rsm_file_index**

```sql
CREATE TABLE rsm_file_index (
    id INT AUTO_INCREMENT PRIMARY KEY,
    wp_attachment_id BIGINT NOT NULL UNIQUE,
    storage_space_id INT NOT NULL,
    relative_path VARCHAR(500) NOT NULL,
    full_url VARCHAR(500) NOT NULL,
    file_size BIGINT NULL,
    mime_type VARCHAR(255) NULL,
    checksum VARCHAR(64) NULL,
    width INT NULL,
    height INT NULL,
    is_thumbnail BOOLEAN DEFAULT FALSE,
    parent_attachment_id BIGINT NULL,
    upload_date TIMESTAMP NULL,
    last_accessed TIMESTAMP NULL,
    access_count INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    FOREIGN KEY (storage_space_id) REFERENCES rsm_storage_spaces(id) ON DELETE CASCADE,
    INDEX idx_attachment (wp_attachment_id),
    INDEX idx_storage (storage_space_id),
    INDEX idx_path (relative_path),
    INDEX idx_thumbnail (is_thumbnail),
    INDEX idx_mime (mime_type)
);
```

### **Field Descriptions**

- **id**: Unique identifier
- **wp_attachment_id**: WordPress attachment ID
- **storage_space_id**: Storage space reference
- **relative_path**: Path within storage
- **full_url**: Complete public URL
- **file_size**: File size in bytes
- **mime_type**: MIME type
- **checksum**: File hash (SHA-256)
- **width**: Image width (pixels)
- **height**: Image height (pixels)
- **is_thumbnail**: Thumbnail flag
- **parent_attachment_id**: Original attachment
- **upload_date**: Original upload timestamp
- **last_accessed**: Last access time
- **access_count**: Access frequency

---

# **5. Queue Table**

## **5.1 rsm_queue**

```sql
CREATE TABLE rsm_queue (
    id INT AUTO_INCREMENT PRIMARY KEY,
    action ENUM('upload', 'delete', 'replace', 'migrate', 'test', 'backup') NOT NULL,
    storage_space_id INT NOT NULL,
    wp_attachment_id BIGINT NULL,
    priority INT DEFAULT 0,
    status ENUM('pending', 'processing', 'completed', 'failed', 'cancelled') DEFAULT 'pending',
    progress INT DEFAULT 0,
    total_size BIGINT NULL,
    processed_size BIGINT NULL,
    error_message TEXT NULL,
    error_code VARCHAR(50) NULL,
    retry_count INT DEFAULT 0,
    max_retries INT DEFAULT 3,
    next_retry_at TIMESTAMP NULL,
    started_at TIMESTAMP NULL,
    completed_at TIMESTAMP NULL,
    created_by BIGINT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    FOREIGN KEY (storage_space_id) REFERENCES rsm_storage_spaces(id) ON DELETE CASCADE,
    INDEX idx_status (status),
    INDEX idx_action (action),
    INDEX idx_priority (priority),
    INDEX idx_retry (next_retry_at),
    INDEX idx_storage_action (storage_space_id, action)
);
```

### **Field Descriptions**

- **id**: Unique identifier
- **action**: Operation type
- **storage_space_id**: Target storage space
- **wp_attachment_id**: Related attachment
- **priority**: Job priority (higher = more important)
- **status**: Current job status
- **progress**: Completion percentage
- **total_size**: Total bytes to process
- **processed_size**: Bytes processed so far
- **error_message**: Error description
- **error_code**: Machine-readable error code
- **retry_count**: Current retry attempt
- **max_retries**: Maximum allowed retries
- **next_retry_at**: Scheduled retry time
- **started_at**: Job start timestamp
- **completed_at**: Job completion timestamp
- **created_by**: User who initiated job

---

# **6. Logs Table**

## **6.1 rsm_logs**

```sql
CREATE TABLE rsm_logs (
    id INT AUTO_INCREMENT PRIMARY KEY,
    action VARCHAR(50) NOT NULL,
    storage_space_id INT NULL,
    wp_attachment_id BIGINT NULL,
    user_id BIGINT NULL,
    ip_address VARCHAR(45) NULL,
    user_agent TEXT NULL,
    details TEXT NULL,
    status ENUM('success', 'error', 'warning', 'info') NOT NULL,
    error_code VARCHAR(50) NULL,
    execution_time DECIMAL(10,4) NULL,
    memory_usage INT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    FOREIGN KEY (storage_space_id) REFERENCES rsm_storage_spaces(id) ON DELETE SET NULL,
    INDEX idx_action (action),
    INDEX idx_status (status),
    INDEX idx_user (user_id),
    INDEX idx_created (created_at),
    INDEX idx_storage_action (storage_space_id, action)
);
```

### **Field Descriptions**

- **id**: Unique identifier
- **action**: Performed action
- **storage_space_id**: Related storage space
- **wp_attachment_id**: Related attachment
- **user_id**: WordPress user ID
- **ip_address**: Client IP address
- **user_agent**: Browser/client string
- **details**: Additional context
- **status**: Log level
- **error_code**: Error identifier
- **execution_time**: Processing time (seconds)
- **memory_usage**: Memory used (bytes)

---

# **7. Settings Table**

## **7.1 rsm_settings**

```sql
CREATE TABLE rsm_settings (
    id INT AUTO_INCREMENT PRIMARY KEY,
    setting_key VARCHAR(100) NOT NULL UNIQUE,
    setting_value LONGTEXT NULL,
    setting_type ENUM('string', 'integer', 'boolean', 'array', 'json') DEFAULT 'string',
    description TEXT NULL,
    is_public BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    INDEX idx_key (setting_key),
    INDEX idx_public (is_public)
);
```

### **Default Settings**

```sql
INSERT INTO rsm_settings (setting_key, setting_value, setting_type, description) VALUES
('version', '1.0.0', 'string', 'Plugin version'),
('default_storage', '0', 'integer', 'Default storage space ID'),
('disable_local', 'false', 'boolean', 'Disable local storage'),
('queue_interval', '60', 'integer', 'Queue processing interval (seconds)'),
('max_file_size', '104857600', 'integer', 'Maximum file size (bytes)'),
('allowed_extensions', 'jpg,jpeg,png,gif,pdf,doc,docx', 'string', 'Allowed file extensions'),
('thumbnail_sizes', '[{"width":150,"height":150,"crop":true}]', 'json', 'Thumbnail sizes'),
('backup_enabled', 'false', 'boolean', 'Enable automatic backups'),
('backup_retention', '30', 'integer', 'Backup retention (days)'),
('log_retention', '90', 'integer', 'Log retention (days)'),
('debug_mode', 'false', 'boolean', 'Enable debug logging');
```

---

# **8. Index Strategy**

## **8.1 Performance Indexes**

```sql
-- File lookup by attachment
CREATE INDEX idx_file_attachment ON rsm_file_index(wp_attachment_id);

-- Queue processing
CREATE INDEX idx_queue_processing ON rsm_queue(status, priority DESC, created_at);

-- Storage space files
CREATE INDEX idx_storage_files ON rsm_file_index(storage_space_id, is_thumbnail);

-- Log queries
CREATE INDEX idx_log_search ON rsm_logs(action, status, created_at);
```

## **8.2 Composite Indexes**

```sql
-- Storage and status
CREATE INDEX idx_storage_status ON rsm_storage_spaces(status, type);

-- File URL lookup
CREATE INDEX idx_file_url ON rsm_file_index(storage_space_id, relative_path);

-- Queue retry management
CREATE INDEX idx_queue_retry ON rsm_queue(status, next_retry_at);
```

---

# **9. Constraints & Triggers**

## **9.1 Foreign Key Constraints**

```sql
-- Cascade delete for storage spaces
ALTER TABLE rsm_file_index 
ADD CONSTRAINT fk_file_storage 
FOREIGN KEY (storage_space_id) REFERENCES rsm_storage_spaces(id) 
ON DELETE CASCADE;

-- Prevent orphaned queue jobs
ALTER TABLE rsm_queue 
ADD CONSTRAINT fk_queue_storage 
FOREIGN KEY (storage_space_id) REFERENCES rsm_storage_spaces(id) 
ON DELETE CASCADE;
```

## **9.2 Check Constraints**

```sql
-- Valid port numbers
ALTER TABLE rsm_storage_spaces 
ADD CONSTRAINT chk_port_range 
CHECK (port BETWEEN 1 AND 65535);

-- Positive file sizes
ALTER TABLE rsm_file_index 
ADD CONSTRAINT chk_file_size 
CHECK (file_size IS NULL OR file_size > 0);

-- Valid progress percentage
ALTER TABLE rsm_queue 
ADD CONSTRAINT chk_progress_range 
CHECK (progress BETWEEN 0 AND 100);
```

## **9.3 Triggers**

```sql
-- Update modified timestamp
DELIMITER //
CREATE TRIGGER tr_storage_updated 
BEFORE UPDATE ON rsm_storage_spaces
FOR EACH ROW
BEGIN
    SET NEW.updated_at = CURRENT_TIMESTAMP;
END//
DELIMITER ;

-- Log storage changes
DELIMITER //
CREATE TRIGGER tr_storage_log 
AFTER UPDATE ON rsm_storage_spaces
FOR EACH ROW
BEGIN
    IF OLD.status != NEW.status THEN
        INSERT INTO rsm_logs (action, storage_space_id, details, status)
        VALUES ('status_change', NEW.id, 
                CONCAT('Status changed from ', OLD.status, ' to ', NEW.status), 
                'info');
    END IF;
END//
DELIMITER ;
```

---

# **10. Data Migration**

## **10.1 Version Upgrade**

```sql
-- v1.0 to v1.1 migration
ALTER TABLE rsm_storage_spaces 
ADD COLUMN ssl_verify BOOLEAN DEFAULT TRUE AFTER passive_mode;

-- v1.1 to v1.2 migration
ALTER TABLE rsm_file_index 
ADD COLUMN access_count INT DEFAULT 0 AFTER last_accessed;
```

## **10.2 Data Import**

```sql
-- Import existing attachments
INSERT INTO rsm_file_index (wp_attachment_id, relative_path, full_url, file_size, mime_type)
SELECT 
    ID,
    SUBSTRING(guid, LENGTH(SUBSTRING_INDEX(guid, '/', -2)) + 1),
    guid,
    meta_value,
    post_mime_type
FROM wp_posts p
JOIN wp_postmeta m ON p.ID = m.post_id
WHERE p.post_type = 'attachment'
AND m.meta_key = '_wp_attachment_metadata';
```

---

# **11. Performance Optimization**

## **11.1 Query Optimization**

```sql
-- Efficient file lookup
SELECT fi.*, ss.name, ss.public_url
FROM rsm_file_index fi
JOIN rsm_storage_spaces ss ON fi.storage_space_id = ss.id
WHERE fi.wp_attachment_id = %d
AND ss.status = 'active';

-- Queue processing
SELECT * FROM rsm_queue 
WHERE status = 'pending' 
AND (next_retry_at IS NULL OR next_retry_at <= NOW())
ORDER BY priority DESC, created_at ASC
LIMIT 10;
```

## **11.2 Caching Strategy**

```sql
-- Storage space cache
CREATE TEMPORARY TABLE temp_storage_cache AS
SELECT * FROM rsm_storage_spaces WHERE status = 'active';

-- File index cache for frequent access
CREATE TEMPORARY TABLE temp_recent_files AS
SELECT * FROM rsm_file_index 
WHERE last_accessed > DATE_SUB(NOW(), INTERVAL 1 HOUR);
```

---

# **12. Backup & Recovery**

## **12.1 Backup Strategy**

```sql
-- Full backup
mysqldump --single-transaction --routines --triggers 
    wordpress_db rsm_storage_spaces rsm_file_index rsm_queue rsm_logs rsm_settings > backup.sql

-- Incremental backup (logs only)
mysqldump --single-transaction --where="created_at > DATE_SUB(NOW(), INTERVAL 1 DAY)"
    wordpress_db rsm_logs > logs_backup.sql
```

## **12.2 Recovery Procedures**

```sql
-- Restore storage spaces
LOCK TABLES rsm_storage_spaces WRITE;
SOURCE backup_storage_spaces.sql;
UNLOCK TABLES;

-- Verify data integrity
SELECT COUNT(*) as total_files, 
       SUM(file_size) as total_size
FROM rsm_file_index 
WHERE storage_space_id = %d;
```

---

# **13. Security Considerations**

## **13.1 Data Protection**

```sql
-- Encrypt sensitive data
ALTER TABLE rsm_storage_spaces 
MODIFY COLUMN password TEXT NOT NULL;

-- Audit trail
CREATE VIEW v_storage_changes AS
SELECT 
    sl.*,
    ss.name as storage_name
FROM rsm_logs sl
JOIN rsm_storage_spaces ss ON sl.storage_space_id = ss.id
WHERE sl.action IN ('create', 'update', 'delete', 'test');
```

## **13.2 Access Control**

```sql
-- Read-only user for monitoring
CREATE USER 'rsm_monitor'@'localhost' IDENTIFIED BY 'secure_password';
GRANT SELECT ON wordpress_db.rsm_storage_spaces TO 'rsm_monitor'@'localhost';
GRANT SELECT ON wordpress_db.rsm_file_index TO 'rsm_monitor'@'localhost';

-- Application user
CREATE USER 'rsm_app'@'localhost' IDENTIFIED BY 'app_password';
GRANT SELECT, INSERT, UPDATE, DELETE ON wordpress_db.rsm_* TO 'rsm_app'@'localhost';
```

---

# **14. Monitoring & Maintenance**

## **14.1 Health Queries**

```sql
-- Storage space status
SELECT 
    name, type, status, 
    TIMESTAMPDIFF(SECOND, last_test, NOW()) as seconds_since_test
FROM rsm_storage_spaces
ORDER BY status, last_test;

-- Queue health
SELECT 
    status, 
    COUNT(*) as count,
    AVG(TIMESTAMPDIFF(SECOND, created_at, COALESCE(completed_at, NOW()))) as avg_duration
FROM rsm_queue
GROUP BY status;

-- Storage usage
SELECT 
    ss.name,
    COUNT(fi.id) as file_count,
    SUM(fi.file_size) as total_size,
    AVG(fi.file_size) as avg_size
FROM rsm_storage_spaces ss
LEFT JOIN rsm_file_index fi ON ss.id = fi.storage_space_id
GROUP BY ss.id, ss.name;
```

## **14.2 Maintenance Tasks**

```sql
-- Clean old logs
DELETE FROM rsm_logs 
WHERE created_at < DATE_SUB(NOW(), INTERVAL 90 DAY);

-- Clean completed queue
DELETE FROM rsm_queue 
WHERE status = 'completed' 
AND completed_at < DATE_SUB(NOW(), INTERVAL 7 DAY);

-- Optimize tables
OPTIMIZE TABLE rsm_storage_spaces;
OPTIMIZE TABLE rsm_file_index;
OPTIMIZE TABLE rsm_queue;
OPTIMIZE TABLE rsm_logs;
```