# **📦 STORAGE SPACES SPECIFICATION**

# **Remote Media Storage**

---

# **1. Overview**

Storage Spaces are the core abstraction layer that allows Remote Media Storage to work with different types of external storage systems while providing a consistent interface.

---

# **2. Storage Space Definition**

A Storage Space represents a configured remote storage location with:

- **Connection Details**: Server credentials and settings
- **Path Configuration**: Root directory and public URL
- **Status Information**: Health and availability
- **Capabilities**: Supported operations and limits

---

# **3. Supported Storage Types**

## **3.1 FTP (File Transfer Protocol)**

### **Configuration**
- Host: Server hostname or IP
- Port: 21 (default)
- Username: FTP account
- Password: FTP password
- Passive Mode: Enabled by default
- Timeout: 30 seconds

### **Capabilities**
- File upload/download
- Directory operations
- File listing
- Size checking

### **Limitations**
- Unencrypted transmission
- Performance overhead
- Firewall issues

## **3.2 SFTP (SSH File Transfer Protocol)**

### **Configuration**
- Host: Server hostname or IP
- Port: 22 (default)
- Username: SSH account
- Password/Key: Authentication
- Timeout: 30 seconds

### **Capabilities**
- Encrypted transmission
- Secure authentication
- File operations
- Directory management

### **Advantages**
- Security
- Reliability
- Firewall friendly

## **3.3 FTPS (FTP over SSL/TLS)**

### **Configuration**
- Host: Server hostname or IP
- Port: 990 (implicit) or 21 (explicit)
- Username: FTP account
- Password: FTP password
- SSL Mode: Implicit or Explicit
- Timeout: 30 seconds

### **Capabilities**
- Encrypted FTP
- Certificate validation
- Secure transfers

### **Considerations**
- Certificate management
- Port configuration
- Compatibility

---

# **4. Storage Space Configuration**

## **4.1 Basic Settings**

```json
{
    "name": "Main Storage",
    "type": "sftp",
    "host": "storage.example.com",
    "port": 22,
    "username": "user",
    "password": "encrypted_password",
    "root_path": "/media/",
    "public_url": "https://cdn.example.com/media"
}
```

## **4.2 Advanced Settings**

```json
{
    "timeout": 30,
    "passive_mode": true,
    "ssl_verify": true,
    "private_key": "path/to/key",
    "max_connections": 5,
    "retry_count": 3,
    "chunk_size": 8192
}
```

## **4.3 Pro Features**

```json
{
    "backup_enabled": true,
    "backup_schedule": "daily",
    "health_check": true,
    "health_interval": 300,
    "quota_limit": 10737418240,
    "quota_warning": 0.8
}
```

---

# **5. Path Management**

## **5.1 Root Path**

- Base directory for all files
- Must exist and be writable
- Can be nested (e.g., `/media/uploads/`)
- Trailing slash normalized

## **5.2 Public URL Structure**

```
Base URL: https://cdn.example.com/media
File Path: 2024/01/image.jpg
Full URL: https://cdn.example.com/media/2024/01/image.jpg
```

## **5.3 Path Sanitization**

- Remove leading slashes
- Encode special characters
- Prevent directory traversal
- Validate file extensions

---

# **6. File Organization**

## **6.1 WordPress Structure**

```
/root_path/
    /2024/
        /01/
            image.jpg
            video.mp4
        /02/
    /2023/
        /12/
```

## **6.2 Custom Structure**

```
/root_path/
    /images/
        /thumbnails/
        /originals/
    /documents/
    /videos/
```

## **6.3 Storage Space Isolation**

- Each space has independent root
- No cross-space file access
- Separate URL namespaces
- Individual permissions

---

# **7. Status Management**

## **7.1 Status Types**

- **Active**: Fully operational
- **Inactive**: Disabled by admin
- **Error**: Connection or permission issues
- **Testing**: Currently being tested
- **Maintenance**: Temporary unavailable

## **7.2 Status Transitions**

```
Created → Testing → Active
Active → Error → Testing → Active
Active → Inactive → Active
Error → Maintenance → Active
```

## **7.3 Health Indicators**

- Connection success rate
- Last successful operation
- Error frequency
- Response times

---

# **8. Security Configuration**

## **8.1 Credential Storage**

```php
// Encryption method
$method = 'AES-256-CBC';
$key = wp_salt('auth');
$iv = openssl_random_pseudo_bytes(16);

// Encrypt password
$encrypted = openssl_encrypt($password, $method, $key, 0, $iv);
$stored = base64_encode($iv . $encrypted);
```

## **8.2 Connection Security**

- Force secure protocols
- Certificate validation
- Host key verification
- Timeout protection

## **8.3 Path Security**

- Whitelist allowed directories
- Prevent symlink attacks
- Validate file permissions
- Audit trail logging

---

# **9. Performance Optimization**

## **9.1 Connection Pooling**

```php
class ConnectionPool {
    private $connections = [];
    private $maxConnections = 5;
    
    public function getConnection($spaceId) {
        if (isset($this->connections[$spaceId])) {
            return $this->connections[$spaceId];
        }
        return $this->createConnection($spaceId);
    }
}
```

## **9.2 Transfer Optimization**

- Parallel uploads
- Chunked transfers
- Compression support
- Caching strategies

## **9.3 Caching Layer**

```php
class StorageCache {
    private $cache = [];
    private $ttl = 300;
    
    public function get($key) {
        if (isset($this->cache[$key])) {
            $item = $this->cache[$key];
            if ($item['expires'] > time()) {
                return $item['data'];
            }
        }
        return null;
    }
}
```

---

# **10. Error Handling**

## **10.1 Connection Errors**

- Host unreachable
- Authentication failure
- Timeout exceeded
- Certificate issues

## **10.2 Operation Errors**

- Permission denied
- Disk full
- File not found
- Network interruption

## **10.3 Recovery Strategies**

```php
class ErrorHandler {
    public function handleConnectionError($space, $error) {
        // Log error
        $this->logError($space, $error);
        
        // Update status
        $this->updateStatus($space, 'error');
        
        // Schedule retry
        $this->scheduleRetry($space);
        
        // Notify admin
        $this->notifyAdmin($space, $error);
    }
}
```

---

# **11. Monitoring & Metrics**

## **11.1 Performance Metrics**

- Transfer speeds
- Connection times
- Success rates
- Error frequencies

## **11.2 Usage Statistics**

- Files stored
- Total size used
- Bandwidth consumed
- Request counts

## **11.3 Health Checks**

```php
class HealthChecker {
    public function checkStorage($space) {
        $results = [
            'connection' => $this->testConnection($space),
            'permissions' => $this->testPermissions($space),
            'space' => $this->checkDiskSpace($space),
            'speed' => $this->testSpeed($space)
        ];
        
        return $this->calculateHealth($results);
    }
}
```

---

# **12. API Interface**

## **12.1 Storage Space Operations**

```php
interface StorageSpaceInterface {
    public function connect(): bool;
    public function disconnect(): void;
    public function testConnection(): array;
    public function uploadFile(string $local, string $remote): bool;
    public function deleteFile(string $remote): bool;
    public function fileExists(string $remote): bool;
    public function getFileSize(string $remote): int;
    public function listFiles(string $directory): array;
}
```

## **12.2 Management Operations**

```php
class StorageSpaceManager {
    public function createSpace(array $config): int;
    public function updateSpace(int $id, array $config): bool;
    public function deleteSpace(int $id): bool;
    public function testSpace(int $id): array;
    public function getSpace(int $id): array;
    public function listSpaces(): array;
}
```

---

# **13. Future Extensions**

## **13.1 Cloud Storage**

- Amazon S3
- Google Cloud Storage
- Azure Blob Storage
- Cloudflare R2

## **13.2 Advanced Features**

- CDN integration
- Automatic scaling
- Geographic distribution
- Smart routing

## **13.3 Enterprise Features**

- SSO integration
- Role-based access
- Audit logging
- Compliance tools

---

# **14. Migration & Backup**

## **14.1 Space Migration**

- Export configuration
- Transfer files
- Update references
- Validate migration

## **14.2 Backup Strategies**

- Configuration backup
- File synchronization
- Point-in-time recovery
- Disaster recovery

---

# **15. Best Practices**

## **15.1 Configuration**

- Use strong passwords
- Enable encryption
- Set appropriate timeouts
- Monitor disk space

## **15.2 Security**

- Regular credential rotation
- SSL certificate validation
- Network segmentation
- Access logging

## **15.3 Performance**

- Optimize connection pooling
- Use appropriate chunk sizes
- Implement caching
- Monitor bottlenecks