<?php
namespace AliKhaleghi\RemoteMediaStorage\Admin;

class AdminInterface {
    public function __construct() {
        add_action('admin_menu', [$this, 'addMenu']);
        add_action('admin_enqueue_scripts', [$this, 'enqueueScripts']);
        add_action('admin_init', [$this, 'handleFormSubmissions']);
        add_action('wp_ajax_rsm_test_connection', [$this, 'handleAjaxTestConnection']);
    }
    
    public function addMenu(): void {
        add_menu_page(
            'Remote Media Storage',
            'Remote Storage',
            'manage_options',
            'remote-media-storage',
            [$this, 'renderDashboard'],
            'dashicons-cloud',
            30
        );
        
        add_submenu_page(
            'remote-media-storage',
            'Storage Spaces',
            'Storage Spaces',
            'manage_options',
            'rsm-storage-spaces',
            [$this, 'renderStorageSpaces']
        );
        
        add_submenu_page(
            'remote-media-storage',
            'Settings',
            'Settings',
            'manage_options',
            'rsm-settings',
            [$this, 'renderSettings']
        );
    }
    
    public function enqueueScripts(string $hook): void {
        if (strpos($hook, 'remote-media-storage') !== false) {
            wp_enqueue_style('rsm-admin', plugins_url('assets/css/admin.css', dirname(__FILE__, 2) . '/remote-media-storage.php'));
            wp_enqueue_script('rsm-admin', plugins_url('assets/js/admin.js', dirname(__FILE__, 2) . '/remote-media-storage.php'), ['jquery'], '0.0.1', true);
            
            wp_localize_script('rsm-admin', 'rsm_admin', [
                'nonce' => wp_create_nonce('rsm_test_connection'),
                'ajax_url' => admin_url('admin-ajax.php')
            ]);
        }
    }
    
    public function renderDashboard(): void {
        ?>
        <div class="wrap">
            <h1>Remote Media Storage</h1>
            <div class="rsm-dashboard">
                <div class="rsm-card">
                    <h2>Overview</h2>
                    <p>Version 0.0.1</p>
                    <p>Store your WordPress media files on external FTP/SFTP servers.</p>
                </div>
                
                <div class="rsm-card">
                    <h2>Quick Actions</h2>
                    <p>
                        <a href="<?php echo admin_url('admin.php?page=rsm-storage-spaces'); ?>" class="button button-primary">
                            Configure Storage Space
                        </a>
                    </p>
                </div>
            </div>
        </div>
        <?php
    }
    
    public function renderStorageSpaces(): void {
        $storageSpaces = $this->getStorageSpaces();
        ?>
        <div class="wrap">
            <h1>Storage Spaces</h1>
            <p>Configure your external storage servers.</p>
            
            <?php if (!empty($storageSpaces)): ?>
                <div class="rsm-existing-storage">
                    <h2>Existing Storage Spaces</h2>
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th>Name</th>
                                <th>Type</th>
                                <th>Host</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($storageSpaces as $storage): ?>
                                <tr>
                                    <td><?php echo esc_html($storage->name); ?></td>
                                    <td><?php echo esc_html(strtoupper($storage->type)); ?></td>
                                    <td><?php echo esc_html($storage->host); ?></td>
                                    <td>
                                        <span class="rsm-status-indicator rsm-status-<?php echo esc_attr($storage->status); ?>"></span>
                                        <?php echo esc_html(ucfirst($storage->status)); ?>
                                    </td>
                                    <td>
                                        <button type="button" class="button rsm-test-connection" data-id="<?php echo esc_attr($storage->id); ?>">
                                            Test Connection
                                        </button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
            
            <div class="rsm-storage-spaces">
                <h2><?php echo empty($storageSpaces) ? 'Add Storage Space' : 'Add New Storage Space'; ?></h2>
                <form method="post" action="">
                    <?php wp_nonce_field('rsm_save_storage', 'rsm_nonce'); ?>
                    <table class="form-table">
                        <tr>
                            <th scope="row">Storage Name</th>
                            <td>
                                <input type="text" name="storage_name" class="regular-text" placeholder="My FTP Server" required>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Type</th>
                            <td>
                                <select name="storage_type">
                                    <option value="ftp">FTP</option>
                                    <option value="sftp">SFTP</option>
                                    <option value="ftps">FTPS</option>
                                </select>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Host</th>
                            <td>
                                <input type="text" name="storage_host" class="regular-text" placeholder="ftp.example.com" required>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Port</th>
                            <td>
                                <input type="number" name="storage_port" value="21" class="small-text">
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Username</th>
                            <td>
                                <input type="text" name="storage_username" class="regular-text" required>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Password</th>
                            <td>
                                <input type="password" name="storage_password" class="regular-text" required>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Root Path</th>
                            <td>
                                <input type="text" name="storage_root" value="/" class="regular-text">
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Public URL</th>
                            <td>
                                <input type="url" name="storage_public_url" class="regular-text" placeholder="https://cdn.example.com" required>
                            </td>
                        </tr>
                    </table>
                    
                    <?php submit_button('Save Storage Space'); ?>
                </form>
            </div>
        </div>
        <?php
    }
    
    public function renderSettings(): void {
        $enabled = get_option('rsm_enabled', false);
        $deleteLocal = get_option('rsm_delete_local', false);
        ?>
        <div class="wrap">
            <h1>Settings</h1>
            <p>Configure Remote Media Storage settings.</p>
            
            <form method="post" action="">
                <?php wp_nonce_field('rsm_save_settings', 'rsm_nonce'); ?>
                <table class="form-table">
                    <tr>
                        <th scope="row">Enable Remote Storage</th>
                        <td>
                            <label>
                                <input type="checkbox" name="rsm_enabled" value="1" <?php checked($enabled); ?>>
                                Store new uploads on remote storage
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Delete Local Files</th>
                        <td>
                            <label>
                                <input type="checkbox" name="rsm_delete_local" value="1" <?php checked($deleteLocal); ?>>
                                Remove local files after successful upload
                            </label>
                        </td>
                    </tr>
                </table>
                
                <?php submit_button('Save Settings'); ?>
            </form>
        </div>
        <?php
    }
    
    public function handleFormSubmissions(): void {
        if (isset($_POST['rsm_nonce']) && wp_verify_nonce($_POST['rsm_nonce'], 'rsm_save_storage')) {
            $this->handleStorageSave();
        }
        
        if (isset($_POST['rsm_nonce']) && wp_verify_nonce($_POST['rsm_nonce'], 'rsm_save_settings')) {
            $this->handleSettingsSave();
        }
    }
    
    private function handleStorageSave(): void {
        global $wpdb;
        
        // Validate required fields
        if (empty($_POST['storage_name']) || empty($_POST['storage_host']) || 
            empty($_POST['storage_username']) || empty($_POST['storage_password']) || 
            empty($_POST['storage_public_url'])) {
            
            add_action('admin_notices', function() {
                echo '<div class="notice notice-error is-dismissible"><p>All fields are required.</p></div>';
            });
            return;
        }
        
        // Validate URL
        if (!filter_var($_POST['storage_public_url'], FILTER_VALIDATE_URL)) {
            add_action('admin_notices', function() {
                echo '<div class="notice notice-error is-dismissible"><p>Public URL is not valid.</p></div>';
            });
            return;
        }
        
        // Test connection before saving
        $config = [
            'host' => sanitize_text_field($_POST['storage_host']),
            'port' => intval($_POST['storage_port']),
            'username' => sanitize_text_field($_POST['storage_username']),
            'password' => sanitize_text_field($_POST['storage_password'])
        ];
        
        $driver = new \AliKhaleghi\RemoteMediaStorage\Drivers\FTP($config);
        $test = $driver->testConnection();
        
        if (!$test['success']) {
            add_action('admin_notices', function() use ($test) {
                echo '<div class="notice notice-error is-dismissible"><p>Connection test failed: ' . esc_html($test['message']) . '</p></div>';
            });
            return;
        }
        
        $data = [
            'name' => sanitize_text_field($_POST['storage_name']),
            'type' => sanitize_text_field($_POST['storage_type']),
            'host' => sanitize_text_field($_POST['storage_host']),
            'port' => intval($_POST['storage_port']),
            'username' => sanitize_text_field($_POST['storage_username']),
            'password' => sanitize_text_field($_POST['storage_password']),
            'root_path' => sanitize_text_field($_POST['storage_root']),
            'public_url' => esc_url_raw($_POST['storage_public_url']),
            'status' => 'active'
        ];
        
        $result = $wpdb->insert($wpdb->prefix . 'rsm_storage_spaces', $data);
        
        if ($result) {
            add_action('admin_notices', function() {
                echo '<div class="notice notice-success is-dismissible"><p>Storage space saved successfully!</p></div>';
            });
        } else {
            add_action('admin_notices', function() {
                echo '<div class="notice notice-error is-dismissible"><p>Error saving storage space.</p></div>';
            });
        }
    }
    
    private function handleSettingsSave(): void {
        $enabled = isset($_POST['rsm_enabled']) ? 1 : 0;
        $deleteLocal = isset($_POST['rsm_delete_local']) ? 1 : 0;
        
        update_option('rsm_enabled', $enabled);
        update_option('rsm_delete_local', $deleteLocal);
        
        add_action('admin_notices', function() {
            echo '<div class="notice notice-success is-dismissible"><p>Settings saved successfully!</p></div>';
        });
    }
    
    private function getStorageSpaces(): array {
        global $wpdb;
        
        return $wpdb->get_results("SELECT * FROM {$wpdb->prefix}rsm_storage_spaces ORDER BY created_at DESC");
    }
    
    public function handleAjaxTestConnection(): void {
        check_ajax_referer('rsm_test_connection', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $storage_id = intval($_POST['storage_id']);
        
        global $wpdb;
        $storage = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}rsm_storage_spaces WHERE id = %d", $storage_id
        ));
        
        if (!$storage) {
            wp_send_json_error(['message' => 'Storage space not found']);
        }
        
        $config = [
            'host' => $storage->host,
            'port' => $storage->port,
            'username' => $storage->username,
            'password' => $storage->password
        ];
        
        $driver = new \AliKhaleghi\RemoteMediaStorage\Drivers\FTP($config);
        $test = $driver->testConnection();
        
        if ($test['success']) {
            wp_send_json_success($test);
        } else {
            wp_send_json_error($test);
        }
    }
}