<?php
namespace AliKhaleghi\RemoteMediaStorage\Drivers;

use AliKhaleghi\RemoteMediaStorage\Drivers\StorageDriverInterface;

class FTP implements StorageDriverInterface {
    private $connection;
    private $config;
    
    public function __construct(array $config) {
        $this->config = array_merge([
            'host' => '',
            'port' => 21,
            'username' => '',
            'password' => '',
            'timeout' => 90,
            'passive' => true
        ], $config);
    }
    
    public function connect(): bool {
        $this->connection = ftp_connect($this->config['host'], $this->config['port'], $this->config['timeout']);
        
        if (!$this->connection) {
            return false;
        }
        
        $login = ftp_login($this->connection, $this->config['username'], $this->config['password']);
        
        if (!$login) {
            ftp_close($this->connection);
            return false;
        }
        
        if ($this->config['passive']) {
            ftp_pasv($this->connection, true);
        }
        
        return true;
    }
    
    public function disconnect(): void {
        if ($this->connection) {
            ftp_close($this->connection);
            $this->connection = null;
        }
    }
    
    public function testConnection(): array {
        $result = [
            'success' => false,
            'message' => '',
            'details' => []
        ];
        
        if (empty($this->config['host'])) {
            $result['message'] = 'Host is required';
            return $result;
        }
        
        if (empty($this->config['username'])) {
            $result['message'] = 'Username is required';
            return $result;
        }
        
        if (!$this->connect()) {
            $result['message'] = 'Connection failed: Could not connect to ' . $this->config['host'] . ':' . $this->config['port'];
            return $result;
        }
        
        $result['success'] = true;
        $result['message'] = 'Connection successful';
        $result['details'] = [
            'host' => $this->config['host'],
            'port' => $this->config['port'],
            'username' => $this->config['username']
        ];
        
        $this->disconnect();
        return $result;
    }
    
    public function uploadFile(string $localPath, string $remote): bool {
        if (!file_exists($localPath)) {
            error_log("RMS: Local file does not exist: $localPath");
            return false;
        }
        
        if (!$this->connect()) {
            error_log("RMS: Failed to connect to FTP server");
            return false;
        }
        
        $dir = dirname($remote);
        if ($dir !== '.' && !$this->createDirectory($dir)) {
            error_log("RMS: Failed to create directory: $dir");
            $this->disconnect();
            return false;
        }
        
        $result = ftp_put($this->connection, $remote, $localPath, FTP_BINARY);
        $this->disconnect();
        
        if (!$result) {
            error_log("RMS: Failed to upload file: $localPath to $remote");
        }
        
        return $result;
    }
    
    public function deleteFile(string $remote): bool {
        if (!$this->connect()) {
            return false;
        }
        
        $result = ftp_delete($this->connection, $remote);
        $this->disconnect();
        
        return $result;
    }
    
    public function fileExists(string $remote): bool {
        if (!$this->connect()) {
            return false;
        }
        
        $result = ftp_size($this->connection, $remote) !== -1;
        $this->disconnect();
        
        return $result;
    }
    
    public function getFileSize(string $remote): int {
        if (!$this->connect()) {
            return 0;
        }
        
        $size = ftp_size($this->connection, $remote);
        $this->disconnect();
        
        return $size === -1 ? 0 : $size;
    }
    
    public function createDirectory(string $path): bool {
        if (!$this->connect()) {
            return false;
        }
        
        $parts = explode('/', trim($path, '/'));
        $current = '';
        
        foreach ($parts as $part) {
            $current .= '/' . $part;
            if (!ftp_chdir($this->connection, $current)) {
                if (!ftp_mkdir($this->connection, $current)) {
                    $this->disconnect();
                    return false;
                }
            }
        }
        
        $this->disconnect();
        return true;
    }
    
    public function listFiles(string $directory): array {
        if (!$this->connect()) {
            return [];
        }
        
        $files = ftp_nlist($this->connection, $directory);
        $this->disconnect();
        
        return is_array($files) ? $files : [];
    }
    
    public function getPermissions(string $path): string {
        if (!$this->connect()) {
            return '';
        }
        
        $perms = ftp_chmod($this->connection, 0, $path);
        $this->disconnect();
        
        return $perms ? decoct($perms) : '';
    }
}