<?php
namespace AliKhaleghi\RemoteMediaStorage\Integration;

class WordPressHooks {
    public function __construct() {
        // No need for plugin path in this class
    }
    
    public function registerHooks(): void {
        add_filter('wp_handle_upload', [$this, 'handleUpload'], 10, 2);
        add_filter('wp_get_attachment_url', [$this, 'rewriteAttachmentUrl'], 10, 2);
        add_action('delete_attachment', [$this, 'handleAttachmentDeletion']);
        add_action('add_attachment', [$this, 'handleAttachmentAdd']);
    }
    
    public function handleUpload(array $upload, string $context): array {
        if (!$this->isRemoteStorageEnabled()) {
            return $upload;
        }
        
        $storage = $this->getActiveStorage();
        if (!$storage || !$storage->isHealthy()) {
            return $upload;
        }
        
        $relativePath = $this->getRelativePath($upload['file']);
        $remotePath = $storage->getRemotePath($relativePath);
        
        if ($storage->uploadFile($upload['file'], $remotePath)) {
            $this->indexFile($upload, $storage, $relativePath);
            
            if ($this->shouldDeleteLocal()) {
                unlink($upload['file']);
                $upload['file'] = $this->createPlaceholder($upload['file']);
            }
        }
        
        return $upload;
    }
    
    public function rewriteAttachmentUrl(string $url, int $attachmentId): string {
        $fileIndex = $this->getFileIndex($attachmentId);
        
        if ($fileIndex) {
            return $fileIndex['full_url'];
        }
        
        return $url;
    }
    
    public function handleAttachmentDeletion(int $attachmentId): void {
        $fileIndex = $this->getFileIndex($attachmentId);
        
        if ($fileIndex) {
            $storage = $this->getStorageById($fileIndex['storage_space_id']);
            if ($storage) {
                $remotePath = $storage->getRemotePath($fileIndex['relative_path']);
                $storage->deleteFile($remotePath);
            }
            
            $this->removeFileIndex($attachmentId);
        }
    }
    
    public function handleAttachmentAdd(int $attachmentId): void {
        $file = get_attached_file($attachmentId);
        
        if ($file && $this->isRemoteStorageEnabled()) {
            $storage = $this->getActiveStorage();
            if ($storage && $storage->isHealthy()) {
                $relativePath = $this->getRelativePath($file);
                $remotePath = $storage->getRemotePath($relativePath);
                
                if ($storage->uploadFile($file, $remotePath)) {
                    $upload = [
                        'file' => $file,
                        'url' => wp_get_attachment_url($attachmentId),
                        'type' => get_post_mime_type($attachmentId)
                    ];
                    
                    $this->indexFile($upload, $storage, $relativePath);
                    
                    if ($this->shouldDeleteLocal()) {
                        unlink($file);
                    }
                }
            }
        }
    }
    
    private function isRemoteStorageEnabled(): bool {
        return get_option('rsm_enabled', false);
    }
    
    private function shouldDeleteLocal(): bool {
        return get_option('rsm_delete_local', false);
    }
    
    private function getActiveStorage(): ?\AliKhaleghi\RemoteMediaStorage\Storage\StorageSpace {
        global $wpdb;
        
        $storage = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}rsm_storage_spaces WHERE status = 'active' LIMIT 1"
        ));
        
        if (!$storage) {
            return null;
        }
        
        $config = [
            'name' => $storage->name,
            'type' => $storage->type,
            'host' => $storage->host,
            'port' => $storage->port,
            'username' => $storage->username,
            'password' => $storage->password,
            'root_path' => $storage->root_path,
            'public_url' => $storage->public_url,
            'status' => $storage->status
        ];
        
        return new \AliKhaleghi\RemoteMediaStorage\Storage\StorageSpace($storage->id, $config);
    }
    
    private function getStorageById(int $id): ?\AliKhaleghi\RemoteMediaStorage\Storage\StorageSpace {
        global $wpdb;
        
        $storage = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}rsm_storage_spaces WHERE id = %d", $id
        ));
        
        if (!$storage) {
            return null;
        }
        
        $config = [
            'name' => $storage->name,
            'type' => $storage->type,
            'host' => $storage->host,
            'port' => $storage->port,
            'username' => $storage->username,
            'password' => $storage->password,
            'root_path' => $storage->root_path,
            'public_url' => $storage->public_url,
            'status' => $storage->status
        ];
        
        return new \AliKhaleghi\RemoteMediaStorage\Storage\StorageSpace($storage->id, $config);
    }
    
    private function getRelativePath(string $fullPath): string {
        $uploadDir = wp_upload_dir();
        $baseDir = $uploadDir['basedir'];
        
        return str_replace($baseDir . '/', '', $fullPath);
    }
    
    private function indexFile(array $upload, \AliKhaleghi\RemoteMediaStorage\Storage\StorageSpace $storage, string $relativePath): void {
        global $wpdb;
        
        $attachmentId = attachment_url_to_postid($upload['url']);
        
        if (!$attachmentId) {
            // Try to get attachment ID from file path
            $uploadDir = wp_upload_dir();
            $relativeFile = str_replace($uploadDir['baseurl'] . '/', '', $upload['url']);
            $attachmentId = $this->getAttachmentIdByFile($relativeFile);
        }
        
        if (!$attachmentId) {
            error_log("RMS: Could not find attachment ID for file: " . $upload['url']);
            return;
        }
        
        $fullUrl = rtrim($storage->getPublicUrl(), '/') . '/' . ltrim($relativePath, '/');
        $fileSize = file_exists($upload['file']) ? filesize($upload['file']) : 0;
        $checksum = file_exists($upload['file']) ? md5_file($upload['file']) : '';
        
        $result = $wpdb->replace(
            $wpdb->prefix . 'rsm_file_index',
            [
                'wp_attachment_id' => $attachmentId,
                'storage_space_id' => $storage->getId(),
                'relative_path' => $relativePath,
                'full_url' => $fullUrl,
                'file_size' => $fileSize,
                'checksum' => $checksum
            ],
            ['%d', '%d', '%s', '%s', '%d', '%s']
        );
        
        if ($result === false) {
            error_log("RMS: Failed to index file for attachment ID: $attachmentId");
        }
    }
    
    private function getAttachmentIdByFile(string $filename): int {
        global $wpdb;
        
        $attachment_id = $wpdb->get_var($wpdb->prepare(
            "SELECT post_id FROM {$wpdb->postmeta} WHERE meta_key = '_wp_attached_file' AND meta_value = %s",
            $filename
        ));
        
        return (int) $attachment_id;
    }
    
    private function getFileIndex(int $attachmentId): ?array {
        global $wpdb;
        
        return $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}rsm_file_index WHERE wp_attachment_id = %d", $attachmentId
        ), ARRAY_A);
    }
    
    private function removeFileIndex(int $attachmentId): void {
        global $wpdb;
        
        $wpdb->delete(
            $wpdb->prefix . 'rsm_file_index',
            ['wp_attachment_id' => $attachmentId],
            ['%d']
        );
    }
    
    private function createPlaceholder(string $originalPath): string {
        $placeholder = $originalPath . '.rsm-placeholder';
        file_put_contents($placeholder, 'Remote Media Storage placeholder');
        return $placeholder;
    }
}