<?php
namespace AliKhaleghi\RemoteMediaStorage\Storage;

use AliKhaleghi\RemoteMediaStorage\Drivers\StorageDriverInterface;

class StorageSpace {
    private $id;
    private $config;
    private $driver;
    private $status;
    
    public function __construct(int $id, array $config) {
        $this->id = $id;
        $this->config = $config;
        $this->status = $config['status'] ?? 'active';
        $this->driver = $this->createDriver();
    }
    
    public function getId(): int {
        return $this->id;
    }
    
    public function getName(): string {
        return $this->config['name'];
    }
    
    public function getType(): string {
        return $this->config['type'];
    }
    
    public function getStatus(): string {
        return $this->status;
    }
    
    public function testConnection(): array {
        if (!$this->driver) {
            return ['success' => false, 'message' => 'Driver not available'];
        }
        
        return $this->driver->testConnection();
    }
    
    public function getPublicUrl(): string {
        return $this->config['public_url'];
    }
    
    public function getRemotePath(string $relative): string {
        $root = rtrim($this->config['root_path'], '/');
        $relative = ltrim($relative, '/');
        return $root . '/' . $relative;
    }
    
    public function isHealthy(): bool {
        return $this->status === 'active' && $this->driver;
    }
    
    public function uploadFile(string $local, string $remote): bool {
        if (!$this->driver) {
            return false;
        }
        
        return $this->driver->uploadFile($local, $remote);
    }
    
    public function deleteFile(string $remote): bool {
        if (!$this->driver) {
            return false;
        }
        
        return $this->driver->deleteFile($remote);
    }
    
    public function fileExists(string $remote): bool {
        if (!$this->driver) {
            return false;
        }
        
        return $this->driver->fileExists($remote);
    }
    
    private function createDriver(): ?StorageDriverInterface {
        $type = $this->config['type'];
        
        switch ($type) {
            case 'ftp':
                return new \AliKhaleghi\RemoteMediaStorage\Drivers\FTP($this->config);
            default:
                return null;
        }
    }
}