<?php
/**
 * Plugin Name: Remote Media Storage
 * Plugin URI: https://github.com/akhaleghi/remote-media-storage
 * Description: Store WordPress media files on external FTP/SFTP servers
 * Version: 0.0.2
 * Author: Ali Khaleghi
 * License: GPL v2 or later
 * Text Domain: remote-media-storage
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.4
 * Requires PHP: 7.4
 */

namespace AliKhaleghi\RemoteMediaStorage;

defined('ABSPATH') || exit;

class RemoteMediaStorage {
    private static $instance = null;
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_action('plugins_loaded', [$this, 'init']);
        register_activation_hook(__FILE__, [$this, 'activate']);
        register_deactivation_hook(__FILE__, [$this, 'deactivate']);
    }
    
    public function init() {
        $this->loadDependencies();
        $this->setupHooks();
        
        if (is_admin()) {
            new Admin\AdminInterface();
        }
    }
    
    private function loadDependencies() {
        require_once plugin_dir_path(__FILE__) . 'includes/drivers/StorageDriverInterface.php';
        require_once plugin_dir_path(__FILE__) . 'includes/drivers/FTP.php';
        require_once plugin_dir_path(__FILE__) . 'includes/storage/StorageSpace.php';
        require_once plugin_dir_path(__FILE__) . 'includes/integration/WordPressHooks.php';
        require_once plugin_dir_path(__FILE__) . 'includes/admin/AdminInterface.php';
    }
    
    private function setupHooks() {
        $hooks = new Integration\WordPressHooks();
        $hooks->registerHooks();
    }
    
    public function activate() {
        $this->createTables();
        flush_rewrite_rules();
    }
    
    public function deactivate() {
        flush_rewrite_rules();
    }
    
    private function createTables() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        // Storage spaces table
        $sql1 = "CREATE TABLE {$wpdb->prefix}rsm_storage_spaces (
            id INT AUTO_INCREMENT PRIMARY KEY,
            name VARCHAR(255) NOT NULL,
            type ENUM('ftp', 'sftp', 'ftps') NOT NULL,
            host VARCHAR(255) NOT NULL,
            port INT NOT NULL DEFAULT 21,
            username VARCHAR(255) NOT NULL,
            password TEXT NOT NULL,
            root_path VARCHAR(255) DEFAULT '/',
            public_url VARCHAR(500) NOT NULL,
            status ENUM('active', 'inactive', 'error') DEFAULT 'active',
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        ) $charset_collate;";
        
        // File index table
        $sql2 = "CREATE TABLE {$wpdb->prefix}rsm_file_index (
            id INT AUTO_INCREMENT PRIMARY KEY,
            wp_attachment_id BIGINT NOT NULL UNIQUE,
            storage_space_id INT NOT NULL,
            relative_path VARCHAR(500) NOT NULL,
            full_url VARCHAR(500) NOT NULL,
            file_size BIGINT,
            checksum VARCHAR(64),
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) $charset_collate;";
        
        // Queue table
        $sql3 = "CREATE TABLE {$wpdb->prefix}rsm_queue (
            id INT AUTO_INCREMENT PRIMARY KEY,
            action ENUM('upload', 'delete', 'replace', 'migrate') NOT NULL,
            storage_space_id INT NOT NULL,
            wp_attachment_id BIGINT,
            status ENUM('pending', 'processing', 'done', 'error') DEFAULT 'pending',
            error_message TEXT,
            retry_count INT DEFAULT 0,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            processed_at TIMESTAMP NULL
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql1);
        dbDelta($sql2);
        dbDelta($sql3);
    }
}

RemoteMediaStorage::getInstance();