<?php
//Define Dirpath for hooks
define("RSL_DIR_PATH", plugin_dir_path(__FILE__));
define("RSSLinksVersion", "1.3.3");
/**
 * Plugin Name: نمایش لینک های RSS
 * Version: 1.3.3
 * Description: دردسترس بودن آدرس های RSS برای همه ی خبرگذاری ها مهم هست، با این افوزنه فقط با چند کلیک تمامی آدرس های فید برای کاربران نمایش داده میشود.
 * Author: علی خالقی
 * Plugin URI: https://khaleghi.info/
 */

if (!class_exists("RSSLinks")) {
    /**
     * RSSLinks Plugin base object
     */
    class RSSLinks
    {
        /**
         * Constructor -- Load Plugin
         */
        public function __construct()
        {
            // Plugin Configurations
            add_action("init", [$this, "setup_actions"]);
        }

        // ----------------------------------------------------------

        /**
         * Plugin Setup
         *
         * @param NULL
         *
         * @return NULL
         */
        public function setup_actions()
        {
            $path = plugin_basename(__FILE__);

            // Add plugin row info
            add_action(
                "after_plugin_row_{$path}",
                [$this, "add_plugin_row_info"],
                10,
                3,
            );

            // Add content filter for RSS page
            add_filter("the_content", [$this, "add_rss_content"]);

            // Check if RSS page exists and create if needed
            $this->ensure_rss_page_exists();
        }

        // ----------------------------------------------------------

        /**
         * Activate callback - install databases
         *
         * @param NULL
         *
         * @return NULL
         */
        public function activate()
        {
            $this->createRSSPage();
        }

        // ----------------------------------------------------------

        /**
         * Deactivate callback - remove databases
         *
         * @param NULL
         *
         * @return NULL
         */
        public function deactivate()
        {
            $this->deleteRSSPage();
        }

        // ----------------------------------------------------------

        /**
         * Create RSS Links Page
         *
         * @param NULL
         *
         * @return NULL
         */
        public function createRSSPage()
        {
            $title =
                "با استفاده از لینک‌های زیر می‌توانید، به نسخه‌های مختلف RSS سایت دسترسی داشته‌باشید";
            $slug = "rss-help";

            $page_args = [
                "post_type" => "page",
                "post_title" => $title,
                "post_content" => "",
                "post_status" => "publish",
                "post_author" => 1,
                "post_name" => $slug,
            ];

            // Check if page already exists
            $existing_page = get_page_by_path($slug);
            if ($existing_page) {
                update_option("RSL_display_page", $existing_page->ID);
                return $existing_page->ID;
            }

            // Create new page
            if (!function_exists("post_exists")) {
                require_once ABSPATH . "wp-admin/includes/post.php";
            }

            $page_id = wp_insert_post($page_args);

            if ($page_id && !is_wp_error($page_id)) {
                update_option("RSL_display_page", $page_id);
            }

            return $page_id;
        }

        // ----------------------------------------------------------

        /**
         * Delete RSS Link Page
         *
         * @param NULL
         *
         * @return NULL
         */
        public function deleteRSSPage()
        {
            $page_id = get_option("RSL_display_page");

            if ($page_id) {
                wp_delete_post($page_id, true);
            }

            delete_option("RSL_display_page");
        }

        /**
         * Add plugin row info in admin
         */
        public function add_plugin_row_info($plugin_file, $plugin_data, $status)
        {
            $rss_page_url = site_url("rss-help");
            $message = sprintf(
                "آدرس صفحه نمایش فید ها: <a target='_blank' href='%s'>%s</a>",
                esc_url($rss_page_url),
                esc_html($rss_page_url),
            );

            echo '<tr class="plugin-update-tr">
			<td colspan="4" class="plugin-update colspanchange">
				<div class="update-message notice inline notice-alt notice-success">
					<p><strong>' .
                $message .
                '</strong></p>
				</div>
			</td>
		</tr>';
        }

        /**
         * Add RSS content to the RSS page
         */
        public function add_rss_content($content)
        {
            if (get_option("RSL_display_page") == get_the_ID()) {

                $categories = get_categories([
                    "orderby" => "name",
                    "order" => "ASC",
                ]);

                ob_start();
                ?>
				<div>
					<table style="width: 100%; border-collapse: collapse;">
						<thead>
							<tr style="background-color: #f1f1f1;">
								<th style='padding: 8px; border: 1px solid #ddd; text-align: right;'>دسته اخبار</th>
								<th style='padding: 8px; border: 1px solid #ddd; text-align: right;'>آدرس RSS</th>
							</tr>
						</thead>
						<tbody>
							<?php foreach ($categories as $category): ?>
								<tr>
									<td style="padding: 8px; border: 1px solid #ddd;"><?php echo esc_html(
             $category->name,
         ); ?></td>
									<td style="padding: 8px; border: 1px solid #ddd; text-align: left;" dir="ltr">
										<a href="<?php echo esc_url(
              get_category_feed_link($category->term_id),
          ); ?>" target="_blank">
											<?php echo esc_html(urldecode(get_category_feed_link($category->term_id))); ?>
										</a>
									</td>
								</tr>
							<?php endforeach; ?>
						</tbody>
					</table>
				</div>
				<?php $content .= ob_get_clean();
            }
            return $content;
        }

        /**
         * Ensure RSS page exists
         */
        private function ensure_rss_page_exists()
        {
            $page_id = get_option("RSL_display_page");
            if (!$page_id || get_post_status($page_id) === false) {
                $this->createRSSPage();
            }
        }
    }

    // Register activation/deactivation hooks outside the class
    register_activation_hook(__FILE__, function () {
        $rss_links = new RSSLinks();
        $rss_links->activate();
    });

    register_deactivation_hook(__FILE__, function () {
        $rss_links = new RSSLinks();
        $rss_links->deactivate();
    });

    // Execute RSSLinks
    $RSSLinks = new RSSLinks();
}
