<?php
/**
 * Lightweight dependency manager for ahur-system plugin
 */

class RSSLinksDependencyManager
{
    private $api_base = "https://wordpress.khaleghi.info/api";
    private $system_plugin_slug = "ahur-system";
    private $system_plugin_file = "ahur-system/ahur-system.php";

    public function __construct()
    {
        add_action("admin_init", [$this, "check_system_plugin"]);
        add_action("admin_notices", [$this, "show_notices"]);
    }

    public function check_system_plugin()
    {
        if (!is_admin()) return;
        
        $status = $this->get_status();
        
        if ($status === 'not_installed') {
            $this->install_plugin();
        } elseif ($status === 'inactive') {
            $this->activate_plugin();
        }
    }

    private function get_status()
    {
        if (is_plugin_active($this->system_plugin_file)) {
            return 'active';
        }
        
        if (file_exists(WP_PLUGIN_DIR . '/' . $this->system_plugin_file)) {
            return 'inactive';
        }
        
        return 'not_installed';
    }

    private function install_plugin()
    {
        $info = $this->get_plugin_info();
        if (!$info || empty($info['download_link'])) return;

        require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
        require_once ABSPATH . 'wp-admin/includes/plugin-install.php';

        $upgrader = new Plugin_Upgrader(new WP_Ajax_Upgrader_Skin());
        $result = $upgrader->install($info['download_link']);

        if (!is_wp_error($result)) {
            $this->activate_plugin();
        }
    }

    private function activate_plugin()
    {
        if (!is_plugin_active($this->system_plugin_file)) {
            activate_plugin($this->system_plugin_file);
        }
    }

    private function get_plugin_info()
    {
        $cache_key = 'ahur_system_info';
        if ($cached = get_transient($cache_key)) {
            return $cached;
        }

        $response = wp_remote_get($this->api_base . "/plugins/info?query=" . $this->system_plugin_slug, [
            'timeout' => 10,
            'headers' => ['Accept' => 'application/json']
        ]);

        if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
            return false;
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);
        if (!$data['success'] || empty($data['data'])) {
            return false;
        }

        $plugin_data = $data['data'][0];
        $info = [
            'name' => $plugin_data['name'],
            'slug' => $plugin_data['slug'],
            'version' => $plugin_data['version'],
            'download_link' => $plugin_data['downloadlink']
        ];

        set_transient($cache_key, $info, 3600);
        return $info;
    }

    public function show_notices()
    {
        $status = $this->get_status();
        
        if ($status === 'not_installed') {
            echo '<div class="notice notice-info is-dismissible"><p><strong>RSSLinks:</strong> Installing required Ahur System plugin...</p></div>';
        } elseif ($status === 'inactive') {
            echo '<div class="notice notice-warning is-dismissible"><p><strong>RSSLinks:</strong> Please <a href="' . admin_url('plugins.php') . '">activate Ahur System</a> plugin.</p></div>';
        }
    }
}

new RSSLinksDependencyManager();