/**
 * WP WordWeaver Admin JavaScript
 * Handles all interactive functionality in the admin interface
 */

(function($) {
    'use strict';

    var WPWordWeaver = {

        /**
         * Initialize the admin functionality
         */
        init: function() {
            this.bindEvents();
            this.initProcessingForm();
            this.initPostMetaBox();
            this.handleLanguageToggle();
        },

        /**
         * Bind event handlers
         */
        bindEvents: function() {
            // Processing form submission
            $('#wp-wordweaver-process-form').on('submit', this.handleFormSubmission.bind(this));

            // Processing type change
            $('#processing_type').on('change', this.handleProcessingTypeChange.bind(this));

            // Preview processing button
            $('#preview-wordweaver-processing').on('click', this.handlePreviewProcessing.bind(this));

            // Post meta box auto-process toggle
            $('input[name="wp_wordweaver_auto_process"]').on('change', this.handleAutoProcessToggle.bind(this));
        },

        /**
         * Initialize processing form
         */
        initProcessingForm: function() {
            // Set initial state
            this.handleProcessingTypeChange();
        },

        /**
         * Initialize post meta box functionality
         */
        initPostMetaBox: function() {
            // Only run on post edit pages
            if (!$('#wp-wordweaver-meta').length) {
                return;
            }

            // Auto-resize textarea
            $('#wp_wordweaver_instructions').on('input', function() {
                this.style.height = 'auto';
                this.style.height = (this.scrollHeight) + 'px';
            });
        },

        /**
         * Handle language toggle for RTL support
         */
        handleLanguageToggle: function() {
            $('select[name="wp_wordweaver_options[language]"]').on('change', function() {
                var selectedLang = $(this).val();

                if (selectedLang === 'fa') {
                    $('body').addClass('rtl');
                    $('html').attr('dir', 'rtl');
                } else {
                    $('body').removeClass('rtl');
                    $('html').attr('dir', 'ltr');
                }
            });
        },

        /**
         * Handle form submission
         */
        handleFormSubmission: function(e) {
            e.preventDefault();

            var $form = $(e.target);
            var $button = $('#process-content-btn');
            var $spinner = $form.find('.spinner');
            var $result = $('#processing-result');

            // Validate form
            if (!this.validateForm($form)) {
                return false;
            }

            // Prepare form data
            var formData = {
                action: 'wp_wordweaver_process_content',
                nonce: wpWordWeaver.nonce,
                post_id: $('#post_id').val(),
                processing_type: $('#processing_type').val(),
                custom_input: $('#custom_input').val(),
                target_language: $('#target_language').val()
            };

            // Show loading state
            $button.prop('disabled', true);
            $spinner.addClass('is-active');
            $result.hide();

            // Send AJAX request
            $.ajax({
                url: wpWordWeaver.ajaxUrl,
                type: 'POST',
                data: formData,
                success: this.handleProcessingSuccess.bind(this),
                error: this.handleProcessingError.bind(this),
                complete: function() {
                    $button.prop('disabled', false);
                    $spinner.removeClass('is-active');
                }
            });
        },

        /**
         * Handle processing type change
         */
        handleProcessingTypeChange: function() {
            var processingType = $('#processing_type').val();
            var $customRow = $('#custom-input-row');
            var $languageRow = $('#target-language-row');

            // Show/hide relevant fields based on processing type
            switch (processingType) {
                case 'custom':
                    $customRow.show();
                    $languageRow.hide();
                    break;

                case 'translate':
                    $customRow.hide();
                    $languageRow.show();
                    break;

                case 'enhance':
                case 'summarize':
                    $customRow.hide();
                    $languageRow.hide();
                    break;

                default:
                    $customRow.show();
                    $languageRow.hide();
            }
        },

        /**
         * Handle preview processing
         */
        handlePreviewProcessing: function(e) {
            e.preventDefault();

            var $button = $(e.target);
            var instructions = $('#wp_wordweaver_instructions').val();

            if (!instructions.trim()) {
                alert(wpWordWeaver.strings.error + ': Please enter processing instructions first.');
                return;
            }

            // Show preview modal (simplified for now)
            var previewHtml = '<div class="wp-wordweaver-preview">' +
                '<h3>Preview Processing Instructions</h3>' +
                '<div class="preview-content">' +
                '<strong>Instructions:</strong><br>' +
                '<em>' + this.escapeHtml(instructions) + '</em>' +
                '</div>' +
                '<p><small>Note: This is a preview of your instructions. Actual AI processing will be implemented in future versions.</small></p>' +
                '</div>';

            // Simple alert for now (can be enhanced with proper modal later)
            if (confirm('Preview Instructions:\n\n' + instructions + '\n\nWould you like to save these instructions?')) {
                // Instructions will be saved when the post is saved
                $button.text('Instructions Ready').prop('disabled', true);
            }
        },

        /**
         * Handle auto-process toggle
         */
        handleAutoProcessToggle: function(e) {
            var $checkbox = $(e.target);
            var $instructions = $('#wp_wordweaver_instructions');

            if ($checkbox.is(':checked')) {
                $instructions.prop('required', true);
                $instructions.closest('p').find('label').append(' <span class="required">*</span>');

                if (!$instructions.val().trim()) {
                    alert('Please enter processing instructions when auto-processing is enabled.');
                    $instructions.focus();
                }
            } else {
                $instructions.prop('required', false);
                $instructions.closest('p').find('.required').remove();
            }
        },

        /**
         * Validate form before submission
         */
        validateForm: function($form) {
            var postId = $('#post_id').val();
            var processingType = $('#processing_type').val();
            var customInput = $('#custom_input').val();

            // Check if post is selected
            if (!postId) {
                alert(wpWordWeaver.strings.error + ': Please select a post to process.');
                $('#post_id').focus();
                return false;
            }

            // Check custom input for custom processing
            if (processingType === 'custom' && !customInput.trim()) {
                alert(wpWordWeaver.strings.error + ': Please enter custom instructions.');
                $('#custom_input').focus();
                return false;
            }

            return true;
        },

        /**
         * Handle successful processing
         */
        handleProcessingSuccess: function(response) {
            var $result = $('#processing-result');
            var data = response.data;

            $result.removeClass('error').addClass('success');

            var html = '<h3><span class="dashicons dashicons-yes-alt"></span> ' +
                      wpWordWeaver.strings.success + '</h3>';

            html += '<p>' + data.message + '</p>';

            if (data.processed_content) {
                html += '<div class="result-content">' +
                       this.escapeHtml(this.truncateContent(data.processed_content, 500)) +
                       '</div>';
            }

            html += '<div class="result-meta">';
            html += '<span><span class="dashicons dashicons-admin-tools"></span> Type: ' +
                   data.processing_type + '</span>';
            html += '<span><span class="dashicons dashicons-clock"></span> ' +
                   'Processed: ' + new Date().toLocaleString() + '</span>';
            html += '</div>';

            $result.html(html).fadeIn();

            // Scroll to result
            $('html, body').animate({
                scrollTop: $result.offset().top - 50
            }, 500);

            // Show success notice
            this.showNotice(data.message, 'success');
        },

        /**
         * Handle processing error
         */
        handleProcessingError: function(xhr, status, error) {
            var $result = $('#processing-result');
            var message = wpWordWeaver.strings.error;

            if (xhr.responseJSON && xhr.responseJSON.data) {
                message = xhr.responseJSON.data;
            } else if (error) {
                message += ': ' + error;
            }

            $result.removeClass('success').addClass('error');

            var html = '<h3><span class="dashicons dashicons-warning"></span> Error</h3>';
            html += '<p>' + message + '</p>';
            html += '<div class="result-meta">';
            html += '<span><span class="dashicons dashicons-clock"></span> ' +
                   'Failed: ' + new Date().toLocaleString() + '</span>';
            html += '</div>';

            $result.html(html).fadeIn();

            // Show error notice
            this.showNotice(message, 'error');
        },

        /**
         * Show admin notice
         */
        showNotice: function(message, type) {
            var $notice = $('<div class="notice notice-' + type + ' is-dismissible">' +
                          '<p>' + message + '</p>' +
                          '<button type="button" class="notice-dismiss">' +
                          '<span class="screen-reader-text">Dismiss this notice.</span>' +
                          '</button></div>');

            $('.wrap h1').after($notice);

            // Auto-dismiss after 5 seconds
            setTimeout(function() {
                $notice.fadeOut(function() {
                    $(this).remove();
                });
            }, 5000);

            // Handle manual dismiss
            $notice.on('click', '.notice-dismiss', function() {
                $notice.fadeOut(function() {
                    $(this).remove();
                });
            });
        },

        /**
         * Escape HTML to prevent XSS
         */
        escapeHtml: function(text) {
            var map = {
                '&': '&amp;',
                '<': '&lt;',
                '>': '&gt;',
                '"': '&quot;',
                "'": '&#039;'
            };

            return text.replace(/[&<>"']/g, function(m) {
                return map[m];
            });
        },

        /**
         * Truncate content for preview
         */
        truncateContent: function(content, maxLength) {
            if (content.length <= maxLength) {
                return content;
            }

            return content.substr(0, maxLength) + '...';
        },

        /**
         * Format content for display
         */
        formatContent: function(content) {
            // Basic formatting for display
            content = this.escapeHtml(content);
            content = content.replace(/\n\n/g, '</p><p>');
            content = content.replace(/\n/g, '<br>');

            return '<p>' + content + '</p>';
        },

        /**
         * Handle keyboard shortcuts
         */
        handleKeyboardShortcuts: function(e) {
            // Ctrl/Cmd + Enter to submit form
            if ((e.ctrlKey || e.metaKey) && e.keyCode === 13) {
                e.preventDefault();
                $('#wp-wordweaver-process-form').trigger('submit');
            }

            // Escape to close modals/notices
            if (e.keyCode === 27) {
                $('.notice-dismiss').trigger('click');
            }
        }
    };

    /**
     * Initialize when document is ready
     */
    $(document).ready(function() {
        WPWordWeaver.init();

        // Bind keyboard shortcuts
        $(document).on('keydown', WPWordWeaver.handleKeyboardShortcuts);

        // Handle WordPress notice dismissals
        $(document).on('click', '.notice-dismiss', function() {
            $(this).closest('.notice').fadeOut();
        });

        // Auto-focus first form field
        $('#post_id').focus();

        // Initialize tooltips if available
        if ($.fn.tooltip) {
            $('[data-toggle="tooltip"]').tooltip();
        }
    });

    /**
     * Handle page unload warning if form has unsaved changes
     */
    $(window).on('beforeunload', function() {
        var $form = $('#wp-wordweaver-process-form');

        if ($form.length && $form.data('changed')) {
            return 'You have unsaved changes. Are you sure you want to leave?';
        }
    });

    /**
     * Track form changes
     */
    $(document).on('input change', '#wp-wordweaver-process-form input, #wp-wordweaver-process-form select, #wp-wordweaver-process-form textarea', function() {
        $('#wp-wordweaver-process-form').data('changed', true);
    });

    /**
     * Clear change tracking on successful submission
     */
    $(document).on('wp-wordweaver-success', function() {
        $('#wp-wordweaver-process-form').data('changed', false);
    });

    /**
     * Make WPWordWeaver globally accessible for extensibility
     */
    window.WPWordWeaver = WPWordWeaver;

})(jQuery);
