<?php
/**
 * Admin functionality for WP WordWeaver
 */

// Prevent direct access
if (!defined("ABSPATH")) {
    exit();
}

class WP_WordWeaver_Admin
{
    private static $instance = null;

    public static function get_instance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct()
    {
        $this->init();
    }

    public function init()
    {
        add_action("admin_menu", [$this, "add_admin_menu"]);
        add_action("admin_init", [$this, "register_settings"]);
        add_action("wp_ajax_wp_wordweaver_process_content", [
            $this,
            "ajax_process_content",
        ]);
        add_action("add_meta_boxes", [$this, "add_post_meta_boxes"]);
        add_action("save_post", [$this, "save_post_meta"]);
    }

    /**
     * Add admin menu
     */
    public function add_admin_menu()
    {
        // Main menu page
        add_menu_page(
            wp_wordweaver_get_translated_plugin_name(),
            __("WordWeaver", "wp-wordweaver"),
            "manage_options",
            "wp-wordweaver",
            [$this, "admin_page"],
            "dashicons-edit-large",
            30,
        );

        // Settings submenu
        add_submenu_page(
            "wp-wordweaver",
            __("Settings", "wp-wordweaver"),
            __("Settings", "wp-wordweaver"),
            "manage_options",
            "wp-wordweaver-settings",
            [$this, "settings_page"],
        );

        // Content Processor submenu
        add_submenu_page(
            "wp-wordweaver",
            __("Content Processor", "wp-wordweaver"),
            __("Content Processor", "wp-wordweaver"),
            "manage_options",
            "wp-wordweaver-processor",
            [$this, "processor_page"],
        );
    }

    /**
     * Register plugin settings
     */
    public function register_settings()
    {
        register_setting("wp_wordweaver_settings", "wp_wordweaver_options", [
            "sanitize_callback" => [$this, "sanitize_options"],
        ]);

        // General Settings Section
        add_settings_section(
            "wp_wordweaver_general",
            __("General Settings", "wp-wordweaver"),
            [$this, "general_section_callback"],
            "wp_wordweaver_settings",
        );

        // Language Setting
        add_settings_field(
            "language",
            __("Language", "wp-wordweaver"),
            [$this, "language_field_callback"],
            "wp_wordweaver_settings",
            "wp_wordweaver_general",
        );

        // Auto Process Setting
        add_settings_field(
            "auto_process",
            __("Auto Process Content", "wp-wordweaver"),
            [$this, "auto_process_field_callback"],
            "wp_wordweaver_settings",
            "wp_wordweaver_general",
        );

        // Plugin Enabled Setting
        add_settings_field(
            "enabled",
            __("Enable Plugin", "wp-wordweaver"),
            [$this, "enabled_field_callback"],
            "wp_wordweaver_settings",
            "wp_wordweaver_general",
        );
    }

    /**
     * Main admin page
     */
    public function admin_page()
    {
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(
                wp_wordweaver_get_translated_plugin_name(),
            ); ?></h1>

            <div class="wp-wordweaver-dashboard">
                <div class="card">
                    <h2><?php printf(
                        __("Welcome to %s", "wp-wordweaver"),
                        wp_wordweaver_get_translated_plugin_name(),
                    ); ?></h2>
                    <p><?php _e(
                        "Transform your content with powerful manipulation tools and AI assistance.",
                        "wp-wordweaver",
                    ); ?></p>

                    <div class="wp-wordweaver-stats">
                        <div class="stat-box">
                            <h3><?php echo $this->get_processed_posts_count(); ?></h3>
                            <p><?php _e(
                                "Posts Processed",
                                "wp-wordweaver",
                            ); ?></p>
                        </div>
                        <div class="stat-box">
                            <h3><?php echo $this->is_plugin_enabled()
                                ? __("Active", "wp-wordweaver")
                                : __("Inactive", "wp-wordweaver"); ?></h3>
                            <p><?php _e(
                                "Plugin Status",
                                "wp-wordweaver",
                            ); ?></p>
                        </div>
                    </div>

                    <div class="wp-wordweaver-actions">
                        <a href="<?php echo admin_url(
                            "admin.php?page=wp-wordweaver-processor",
                        ); ?>" class="button button-primary">
                            <?php _e("Process Content", "wp-wordweaver"); ?>
                        </a>
                        <a href="<?php echo admin_url(
                            "admin.php?page=wp-wordweaver-settings",
                        ); ?>" class="button">
                            <?php _e("Settings", "wp-wordweaver"); ?>
                        </a>
                    </div>

                    <?php if (defined("WP_DEBUG") && WP_DEBUG): ?>
                    <div class="card" style="margin-top: 20px; background: #f9f9f9;">
                        <h3>Translation Debug Info</h3>
                        <table class="widefat">
                            <tr>
                                <td><strong>Current Locale:</strong></td>
                                <td><?php echo get_locale(); ?></td>
                            </tr>
                            <tr>
                                <td><strong>Text Domain Loaded:</strong></td>
                                <td><?php echo is_textdomain_loaded(
                                    "wp-wordweaver",
                                )
                                    ? "Yes"
                                    : "No"; ?></td>
                            </tr>
                            <tr>
                                <td><strong>Plugin Name (function):</strong></td>
                                <td><?php echo wp_wordweaver_get_plugin_name(); ?></td>
                            </tr>
                            <tr>
                                <td><strong>Plugin Name (translated):</strong></td>
                                <td><?php echo wp_wordweaver_get_translated_plugin_name(); ?></td>
                            </tr>
                            <tr>
                                <td><strong>Sample Translation:</strong></td>
                                <td><?php _e(
                                    "Settings",
                                    "wp-wordweaver",
                                ); ?></td>
                            </tr>
                            <tr>
                                <td><strong>MO File (fa):</strong></td>
                                <td><?php echo file_exists(
                                    WP_WORDWEAVER_PLUGIN_DIR .
                                        "languages/wp-wordweaver-fa.mo",
                                )
                                    ? "Exists"
                                    : "Missing"; ?></td>
                            </tr>
                            <tr>
                                <td><strong>MO File (fa_IR):</strong></td>
                                <td><?php echo file_exists(
                                    WP_WORDWEAVER_PLUGIN_DIR .
                                        "languages/wp-wordweaver-fa_IR.mo",
                                )
                                    ? "Exists"
                                    : "Missing"; ?></td>
                            </tr>
                        </table>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Settings page
     */
    public function settings_page()
    {
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>

            <form method="post" action="options.php">
                <?php
                settings_fields("wp_wordweaver_settings");
                do_settings_sections("wp_wordweaver_settings");
                submit_button();?>
            </form>
        </div>
        <?php
    }

    /**
     * Content processor page
     */
    public function processor_page()
    {
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>

            <div class="wp-wordweaver-processor">
                <div class="card">
                    <h2><?php _e(
                        "Manual Content Processing",
                        "wp-wordweaver",
                    ); ?></h2>

                    <form id="wp-wordweaver-process-form" method="post">
                        <?php wp_nonce_field(
                            "wp_wordweaver_process",
                            "wp_wordweaver_nonce",
                        ); ?>

                        <table class="form-table">
                            <tr>
                                <th scope="row">
                                    <label for="post_id"><?php _e(
                                        "Select Post",
                                        "wp-wordweaver",
                                    ); ?></label>
                                </th>
                                <td>
                                    <?php $posts = get_posts([
                                        "numberposts" => -1,
                                        "post_status" => "any",
                                    ]); ?>
                                    <select name="post_id" id="post_id" class="regular-text">
                                        <option value=""><?php _e(
                                            "Select a post...",
                                            "wp-wordweaver",
                                        ); ?></option>
                                        <?php foreach ($posts as $post): ?>
                                            <option value="<?php echo $post->ID; ?>"><?php echo esc_html(
    $post->post_title,
); ?> (ID: <?php echo $post->ID; ?>)</option>
                                        <?php endforeach; ?>
                                    </select>
                                </td>
                            </tr>

                            <tr>
                                <th scope="row">
                                    <label for="processing_type"><?php _e(
                                        "Processing Type",
                                        "wp-wordweaver",
                                    ); ?></label>
                                </th>
                                <td>
                                    <select name="processing_type" id="processing_type" class="regular-text">
                                        <option value="custom"><?php _e(
                                            "Custom Input",
                                            "wp-wordweaver",
                                        ); ?></option>
                                        <option value="enhance"><?php _e(
                                            "Enhance Content",
                                            "wp-wordweaver",
                                        ); ?></option>
                                        <option value="summarize"><?php _e(
                                            "Summarize",
                                            "wp-wordweaver",
                                        ); ?></option>
                                        <option value="translate"><?php _e(
                                            "Translate",
                                            "wp-wordweaver",
                                        ); ?></option>
                                    </select>
                                </td>
                            </tr>

                            <tr id="custom-input-row">
                                <th scope="row">
                                    <label for="custom_input"><?php _e(
                                        "Custom Instructions",
                                        "wp-wordweaver",
                                    ); ?></label>
                                </th>
                                <td>
                                    <textarea name="custom_input" id="custom_input" rows="5" cols="50" class="large-text" placeholder="<?php _e(
                                        "Enter your custom instructions for content modification...",
                                        "wp-wordweaver",
                                    ); ?>"></textarea>
                                    <p class="description"><?php _e(
                                        "Describe how you want the content to be modified.",
                                        "wp-wordweaver",
                                    ); ?></p>
                                </td>
                            </tr>

                            <tr id="target-language-row" style="display: none;">
                                <th scope="row">
                                    <label for="target_language"><?php _e(
                                        "Target Language",
                                        "wp-wordweaver",
                                    ); ?></label>
                                </th>
                                <td>
                                    <select name="target_language" id="target_language" class="regular-text">
                                        <option value="en"><?php _e(
                                            "English",
                                            "wp-wordweaver",
                                        ); ?></option>
                                        <option value="fa"><?php _e(
                                            "Persian/Farsi",
                                            "wp-wordweaver",
                                        ); ?></option>
                                    </select>
                                </td>
                            </tr>
                        </table>

                        <p class="submit">
                            <button type="submit" class="button button-primary" id="process-content-btn">
                                <?php _e("Process Content", "wp-wordweaver"); ?>
                            </button>
                            <span class="spinner"></span>
                        </p>
                    </form>

                    <div id="processing-result" style="display: none;"></div>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Add meta boxes to post edit screens
     */
    public function add_post_meta_boxes()
    {
        add_meta_box(
            "wp-wordweaver-meta",
            __("WordWeaver Options", "wp-wordweaver"),
            [$this, "post_meta_box_callback"],
            ["post", "page"],
            "side",
            "high",
        );
    }

    /**
     * Post meta box callback
     */
    public function post_meta_box_callback($post)
    {
        wp_nonce_field("wp_wordweaver_post_meta", "wp_wordweaver_post_nonce");

        $auto_process = get_post_meta(
            $post->ID,
            "_wp_wordweaver_auto_process",
            true,
        );
        $custom_instructions = get_post_meta(
            $post->ID,
            "_wp_wordweaver_instructions",
            true,
        );
        ?>

        <p>
            <label>
                <input type="checkbox" name="wp_wordweaver_auto_process" value="1" <?php checked(
                    $auto_process,
                    "1",
                ); ?>>
                <?php _e("Auto-process this post", "wp-wordweaver"); ?>
            </label>
        </p>

        <p>
            <label for="wp_wordweaver_instructions"><?php _e(
                "Custom Instructions:",
                "wp-wordweaver",
            ); ?></label>
            <textarea name="wp_wordweaver_instructions" id="wp_wordweaver_instructions" rows="3" style="width: 100%;" placeholder="<?php _e(
                "Enter custom processing instructions...",
                "wp-wordweaver",
            ); ?>"><?php echo esc_textarea($custom_instructions); ?></textarea>
        </p>

        <p>
            <button type="button" class="button button-secondary" id="preview-wordweaver-processing">
                <?php _e("Preview Processing", "wp-wordweaver"); ?>
            </button>
        </p>
        <?php
    }

    /**
     * Save post meta
     */
    public function save_post_meta($post_id)
    {
        if (
            !isset($_POST["wp_wordweaver_post_nonce"]) ||
            !wp_verify_nonce(
                $_POST["wp_wordweaver_post_nonce"],
                "wp_wordweaver_post_meta",
            )
        ) {
            return;
        }

        if (defined("DOING_AUTOSAVE") && DOING_AUTOSAVE) {
            return;
        }

        if (!current_user_can("edit_post", $post_id)) {
            return;
        }

        $auto_process = isset($_POST["wp_wordweaver_auto_process"]) ? "1" : "0";
        update_post_meta(
            $post_id,
            "_wp_wordweaver_auto_process",
            $auto_process,
        );

        $instructions = sanitize_textarea_field(
            $_POST["wp_wordweaver_instructions"],
        );
        update_post_meta(
            $post_id,
            "_wp_wordweaver_instructions",
            $instructions,
        );
    }

    /**
     * AJAX handler for content processing
     */
    public function ajax_process_content()
    {
        check_ajax_referer("wp_wordweaver_nonce", "nonce");

        if (!current_user_can("manage_options")) {
            wp_die(__("Unauthorized", "wp-wordweaver"));
        }

        $post_id = intval($_POST["post_id"]);
        $processing_type = sanitize_text_field($_POST["processing_type"]);
        $custom_input = sanitize_textarea_field($_POST["custom_input"]);
        $target_language = sanitize_text_field($_POST["target_language"]);

        if (!$post_id) {
            wp_send_json_error(__("Invalid post ID", "wp-wordweaver"));
        }

        // Get the post
        $post = get_post($post_id);
        if (!$post) {
            wp_send_json_error(__("Post not found", "wp-wordweaver"));
        }

        // Process the content using the processor class
        $processor = WP_WordWeaver_Processor::get_instance();
        $result = $processor->process_content(
            $post,
            $processing_type,
            $custom_input,
            $target_language,
        );

        if ($result["success"]) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result["message"]);
        }
    }

    // Settings callbacks
    public function general_section_callback()
    {
        echo "<p>" .
            sprintf(
                __("Configure general settings for %s.", "wp-wordweaver"),
                wp_wordweaver_get_translated_plugin_name(),
            ) .
            "</p>";
    }

    public function language_field_callback()
    {
        $options = get_option("wp_wordweaver_options");
        $language = isset($options["language"]) ? $options["language"] : "en";
        ?>
        <select name="wp_wordweaver_options[language]">
            <option value="en" <?php selected($language, "en"); ?>><?php _e(
    "English",
    "wp-wordweaver",
); ?></option>
            <option value="fa" <?php selected($language, "fa"); ?>><?php _e(
    "Persian/Farsi",
    "wp-wordweaver",
); ?></option>
        </select>
        <?php
    }

    public function auto_process_field_callback()
    {
        $options = get_option("wp_wordweaver_options");
        $auto_process = isset($options["auto_process"])
            ? $options["auto_process"]
            : false;
        ?>
        <label>
            <input type="checkbox" name="wp_wordweaver_options[auto_process]" value="1" <?php checked(
                $auto_process,
                1,
            ); ?>>
            <?php _e(
                "Automatically process content when posts are saved",
                "wp-wordweaver",
            ); ?>
        </label>
        <?php
    }

    public function enabled_field_callback()
    {
        $options = get_option("wp_wordweaver_options");
        $enabled = isset($options["enabled"]) ? $options["enabled"] : true;
        ?>
        <label>
            <input type="checkbox" name="wp_wordweaver_options[enabled]" value="1" <?php checked(
                $enabled,
                1,
            ); ?>>
            <?php _e("Enable WP WordWeaver functionality", "wp-wordweaver"); ?>
        </label>
        <?php
    }

    public function sanitize_options($input)
    {
        $sanitized = [];

        if (isset($input["language"])) {
            $sanitized["language"] = sanitize_text_field($input["language"]);
        }

        $sanitized["auto_process"] = isset($input["auto_process"]) ? 1 : 0;
        $sanitized["enabled"] = isset($input["enabled"]) ? 1 : 0;

        return $sanitized;
    }

    // Helper methods
    private function get_processed_posts_count()
    {
        global $wpdb;
        $count = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->postmeta} WHERE meta_key = '_wp_wordweaver_processed'",
        );
        return intval($count);
    }

    private function is_plugin_enabled()
    {
        $options = get_option("wp_wordweaver_options");
        return isset($options["enabled"]) ? $options["enabled"] : true;
    }
}
