<?php
/**
 * Content processor for WP WordWeaver
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class WP_WordWeaver_Processor {

    private static $instance = null;

    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        $this->init();
    }

    public function init() {
        // Hook into post save if auto-processing is enabled
        add_action('save_post', array($this, 'maybe_auto_process'), 20);
    }

    /**
     * Process content based on admin input
     */
    public function process_content($post, $processing_type, $custom_input = '', $target_language = 'en') {
        if (!$post || !is_object($post)) {
            return array(
                'success' => false,
                'message' => __('Invalid post object', 'wp-wordweaver')
            );
        }

        // Get current content
        $original_content = $post->post_content;
        $processed_content = '';

        // Process based on type
        switch ($processing_type) {
            case 'custom':
                $processed_content = $this->process_custom($original_content, $custom_input);
                break;

            case 'enhance':
                $processed_content = $this->enhance_content($original_content);
                break;

            case 'summarize':
                $processed_content = $this->summarize_content($original_content);
                break;

            case 'translate':
                $processed_content = $this->translate_content($original_content, $target_language);
                break;

            default:
                return array(
                    'success' => false,
                    'message' => __('Invalid processing type', 'wp-wordweaver')
                );
        }

        // Update the post
        $updated_post = array(
            'ID' => $post->ID,
            'post_content' => $processed_content
        );

        $result = wp_update_post($updated_post);

        if (is_wp_error($result)) {
            return array(
                'success' => false,
                'message' => $result->get_error_message()
            );
        }

        // Mark as processed
        update_post_meta($post->ID, '_wp_wordweaver_processed', current_time('mysql'));
        update_post_meta($post->ID, '_wp_wordweaver_original_content', $original_content);
        update_post_meta($post->ID, '_wp_wordweaver_processing_type', $processing_type);

        return array(
            'success' => true,
            'message' => __('Content processed successfully', 'wp-wordweaver'),
            'original_content' => $original_content,
            'processed_content' => $processed_content,
            'processing_type' => $processing_type
        );
    }

    /**
     * Process content with custom instructions
     */
    private function process_custom($content, $instructions) {
        if (empty($instructions)) {
            return $content;
        }

        // For now, we'll simulate processing by adding admin instructions as a note
        // This will be replaced with AI processing later
        $processed = $content . "\n\n<!-- WordWeaver Custom Processing -->\n";
        $processed .= "<!-- Instructions: " . esc_html($instructions) . " -->\n";
        $processed .= "<!-- Note: This is a placeholder. AI processing will be implemented soon. -->";

        return $processed;
    }

    /**
     * Enhance content (placeholder for AI enhancement)
     */
    private function enhance_content($content) {
        // Placeholder enhancement - add some formatting improvements
        $enhanced = $content;

        // Add paragraph breaks for better readability
        $enhanced = wpautop($enhanced);

        // Add enhancement note
        $enhanced .= "\n\n<!-- WordWeaver Enhanced Content -->";

        return $enhanced;
    }

    /**
     * Summarize content
     */
    private function summarize_content($content) {
        // Simple summarization - take first paragraph for now
        $paragraphs = explode("\n\n", $content);
        $summary = '';

        if (!empty($paragraphs)) {
            $summary = $paragraphs[0];
            if (strlen($summary) > 300) {
                $summary = wp_trim_words($summary, 50, '...');
            }
        }

        $summary .= "\n\n<!-- WordWeaver Summary -->";
        $summary .= "\n<!-- Original content length: " . strlen($content) . " characters -->";

        return $summary;
    }

    /**
     * Translate content
     */
    private function translate_content($content, $target_language) {
        // Placeholder translation function
        $translated = $content;

        // Add translation note
        $translated .= "\n\n<!-- WordWeaver Translation -->";
        $translated .= "\n<!-- Target Language: " . $target_language . " -->";
        $translated .= "\n<!-- Note: AI translation will be implemented soon. -->";

        return $translated;
    }

    /**
     * Maybe auto-process post on save
     */
    public function maybe_auto_process($post_id) {
        // Skip if auto-save
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        // Skip if user can't edit posts
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }

        // Check if auto-processing is enabled for this post
        $auto_process = get_post_meta($post_id, '_wp_wordweaver_auto_process', true);
        if (!$auto_process) {
            return;
        }

        // Get custom instructions
        $instructions = get_post_meta($post_id, '_wp_wordweaver_instructions', true);

        // Get the post
        $post = get_post($post_id);
        if (!$post) {
            return;
        }

        // Process the content
        $this->process_content($post, 'custom', $instructions);
    }

    /**
     * Revert content to original
     */
    public function revert_content($post_id) {
        $original_content = get_post_meta($post_id, '_wp_wordweaver_original_content', true);

        if (!$original_content) {
            return array(
                'success' => false,
                'message' => __('No original content found', 'wp-wordweaver')
            );
        }

        $updated_post = array(
            'ID' => $post_id,
            'post_content' => $original_content
        );

        $result = wp_update_post($updated_post);

        if (is_wp_error($result)) {
            return array(
                'success' => false,
                'message' => $result->get_error_message()
            );
        }

        // Clean up meta data
        delete_post_meta($post_id, '_wp_wordweaver_processed');
        delete_post_meta($post_id, '_wp_wordweaver_original_content');
        delete_post_meta($post_id, '_wp_wordweaver_processing_type');

        return array(
            'success' => true,
            'message' => __('Content reverted successfully', 'wp-wordweaver')
        );
    }

    /**
     * Get processing history for a post
     */
    public function get_processing_history($post_id) {
        $processed_date = get_post_meta($post_id, '_wp_wordweaver_processed', true);
        $processing_type = get_post_meta($post_id, '_wp_wordweaver_processing_type', true);
        $original_content = get_post_meta($post_id, '_wp_wordweaver_original_content', true);

        if (!$processed_date) {
            return false;
        }

        return array(
            'processed_date' => $processed_date,
            'processing_type' => $processing_type,
            'has_original' => !empty($original_content)
        );
    }

    /**
     * Check if post has been processed
     */
    public function is_post_processed($post_id) {
        $processed = get_post_meta($post_id, '_wp_wordweaver_processed', true);
        return !empty($processed);
    }

    /**
     * Get all processed posts
     */
    public function get_processed_posts($limit = 10, $offset = 0) {
        global $wpdb;

        $query = $wpdb->prepare("
            SELECT p.*, pm.meta_value as processed_date, pm2.meta_value as processing_type
            FROM {$wpdb->posts} p
            LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_wp_wordweaver_processed'
            LEFT JOIN {$wpdb->postmeta} pm2 ON p.ID = pm2.post_id AND pm2.meta_key = '_wp_wordweaver_processing_type'
            WHERE pm.meta_value IS NOT NULL
            ORDER BY pm.meta_value DESC
            LIMIT %d OFFSET %d
        ", $limit, $offset);

        return $wpdb->get_results($query);
    }

    /**
     * Clean up all processing data
     */
    public function cleanup_all_processing_data() {
        global $wpdb;

        // Delete all WordWeaver meta data
        $wpdb->delete($wpdb->postmeta, array(
            'meta_key' => '_wp_wordweaver_processed'
        ));

        $wpdb->delete($wpdb->postmeta, array(
            'meta_key' => '_wp_wordweaver_original_content'
        ));

        $wpdb->delete($wpdb->postmeta, array(
            'meta_key' => '_wp_wordweaver_processing_type'
        ));

        $wpdb->delete($wpdb->postmeta, array(
            'meta_key' => '_wp_wordweaver_auto_process'
        ));

        $wpdb->delete($wpdb->postmeta, array(
            'meta_key' => '_wp_wordweaver_instructions'
        ));

        return array(
            'success' => true,
            'message' => __('All processing data cleaned up successfully', 'wp-wordweaver')
        );
    }
}
