<?php
/**
 * Helper functions for WP WordWeaver
 */

// Prevent direct access
if (!defined("ABSPATH")) {
    exit();
}

/**
 * Get plugin option with default fallback
 */
function wp_wordweaver_get_option($key, $default = null)
{
    $options = get_option("wp_wordweaver_options", []);
    return isset($options[$key]) ? $options[$key] : $default;
}

/**
 * Update plugin option
 */
function wp_wordweaver_update_option($key, $value)
{
    $options = get_option("wp_wordweaver_options", []);
    $options[$key] = $value;
    return update_option("wp_wordweaver_options", $options);
}

/**
 * Check if plugin is enabled
 */
function wp_wordweaver_is_enabled()
{
    return wp_wordweaver_get_option("enabled", true);
}

/**
 * Get current plugin language
 */
function wp_wordweaver_get_language()
{
    return wp_wordweaver_get_option("language", "en");
}

/**
 * Check if current language is RTL
 */
function wp_wordweaver_is_rtl()
{
    return wp_wordweaver_get_language() === "fa";
}

/**
 * Get localized plugin name
 */
function wp_wordweaver_get_plugin_name()
{
    $locale = get_locale();

    // Check if Persian locale
    if (strpos($locale, "fa") === 0) {
        return "واژه‌باف";
    }

    return "WP WordWeaver";
}

/**
 * Get translated plugin name for current locale
 */
function wp_wordweaver_get_translated_plugin_name()
{
    // Use WordPress translation system
    return __("WP WordWeaver", "wp-wordweaver");
}

/**
 * Format date based on current language
 */
function wp_wordweaver_format_date($date, $format = null)
{
    if (!$format) {
        $format = wp_wordweaver_is_rtl() ? "Y/m/d H:i" : "M j, Y g:i A";
    }

    return date_i18n($format, strtotime($date));
}

/**
 * Sanitize content for processing
 */
function wp_wordweaver_sanitize_content($content)
{
    // Remove potentially harmful content
    $content = wp_kses_post($content);

    // Normalize whitespace
    $content = preg_replace("/\s+/", " ", $content);
    $content = trim($content);

    return $content;
}

/**
 * Truncate text with proper encoding support
 */
function wp_wordweaver_truncate_text($text, $length = 100, $suffix = "...")
{
    if (function_exists("mb_strlen") && mb_strlen($text, "UTF-8") > $length) {
        return mb_substr($text, 0, $length, "UTF-8") . $suffix;
    } elseif (strlen($text) > $length) {
        return substr($text, 0, $length) . $suffix;
    }

    return $text;
}

/**
 * Count words in text (supports Persian/Arabic)
 */
function wp_wordweaver_count_words($text)
{
    // Remove HTML tags
    $text = wp_strip_all_tags($text);

    // For Persian/Arabic text
    if (wp_wordweaver_is_rtl()) {
        // Remove extra whitespace and split by spaces
        $text = preg_replace("/\s+/", " ", trim($text));
        $words = explode(" ", $text);
        return count(array_filter($words));
    }

    // For English text
    return str_word_count($text);
}

/**
 * Estimate reading time
 */
function wp_wordweaver_estimate_reading_time($text, $wpm = null)
{
    if (!$wpm) {
        // Default words per minute (Persian is typically slower)
        $wpm = wp_wordweaver_is_rtl() ? 150 : 200;
    }

    $word_count = wp_wordweaver_count_words($text);
    $minutes = ceil($word_count / $wpm);

    return max(1, $minutes);
}

/**
 * Generate admin notice
 */
function wp_wordweaver_admin_notice(
    $message,
    $type = "success",
    $dismissible = true,
) {
    $dismissible_class = $dismissible ? "is-dismissible" : "";

    return sprintf(
        '<div class="notice notice-%s %s"><p>%s</p></div>',
        esc_attr($type),
        $dismissible_class,
        $message,
    );
}

/**
 * Log debug information
 */
function wp_wordweaver_log($message, $level = "info")
{
    if (defined("WP_DEBUG") && WP_DEBUG) {
        error_log(
            sprintf("[WP WordWeaver] [%s] %s", strtoupper($level), $message),
        );
    }
}

/**
 * Get post types that support WordWeaver
 */
function wp_wordweaver_get_supported_post_types()
{
    $post_types = get_post_types(["public" => true], "objects");

    // Filter out unsupported post types
    unset($post_types["attachment"]);

    return apply_filters("wp_wordweaver_supported_post_types", $post_types);
}

/**
 * Check if post type supports WordWeaver
 */
function wp_wordweaver_post_type_supported($post_type)
{
    $supported_types = wp_wordweaver_get_supported_post_types();
    return isset($supported_types[$post_type]);
}

/**
 * Get processing statistics
 */
function wp_wordweaver_get_stats()
{
    global $wpdb;

    $stats = [];

    // Total processed posts
    $stats["total_processed"] = $wpdb->get_var(
        "SELECT COUNT(*) FROM {$wpdb->postmeta} WHERE meta_key = '_wp_wordweaver_processed'",
    );

    // Processing by type
    $type_stats = $wpdb->get_results(
        "SELECT meta_value as type, COUNT(*) as count
         FROM {$wpdb->postmeta}
         WHERE meta_key = '_wp_wordweaver_processing_type'
         GROUP BY meta_value",
    );

    $stats["by_type"] = [];
    foreach ($type_stats as $stat) {
        $stats["by_type"][$stat->type] = intval($stat->count);
    }

    // Recent processing (last 30 days)
    $stats["recent_processed"] = $wpdb->get_var(
        $wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->postmeta}
         WHERE meta_key = '_wp_wordweaver_processed'
         AND meta_value >= %s",
            date("Y-m-d H:i:s", strtotime("-30 days")),
        ),
    );

    return $stats;
}

/**
 * Generate nonce for WordWeaver actions
 */
function wp_wordweaver_generate_nonce($action = "wp_wordweaver_action")
{
    return wp_create_nonce($action);
}

/**
 * Verify nonce for WordWeaver actions
 */
function wp_wordweaver_verify_nonce($nonce, $action = "wp_wordweaver_action")
{
    return wp_verify_nonce($nonce, $action);
}

/**
 * Get content excerpt for preview
 */
function wp_wordweaver_get_content_excerpt($content, $length = 150)
{
    $excerpt = wp_strip_all_tags($content);
    $excerpt = wp_wordweaver_truncate_text($excerpt, $length);

    return $excerpt;
}

/**
 * Check if content has been modified by WordWeaver
 */
function wp_wordweaver_is_content_modified($post_id)
{
    return !empty(get_post_meta($post_id, "_wp_wordweaver_processed", true));
}

/**
 * Get content modification date
 */
function wp_wordweaver_get_modification_date($post_id)
{
    $date = get_post_meta($post_id, "_wp_wordweaver_processed", true);

    if ($date) {
        return wp_wordweaver_format_date($date);
    }

    return null;
}

/**
 * Clean up content for display
 */
function wp_wordweaver_clean_display_content($content)
{
    // Remove WordWeaver HTML comments
    $content = preg_replace("/<!--\s*WordWeaver.*?-->/s", "", $content);

    return $content;
}

/**
 * Get available processing types
 */
function wp_wordweaver_get_processing_types()
{
    $types = [
        "custom" => __("Custom Input", "wp-wordweaver"),
        "enhance" => __("Enhance Content", "wp-wordweaver"),
        "summarize" => __("Summarize", "wp-wordweaver"),
        "translate" => __("Translate", "wp-wordweaver"),
    ];

    return apply_filters("wp_wordweaver_processing_types", $types);
}

/**
 * Get supported languages
 */
function wp_wordweaver_get_supported_languages()
{
    $languages = [
        "en" => __("English", "wp-wordweaver"),
        "fa" => __("Persian/Farsi", "wp-wordweaver"),
    ];

    return apply_filters("wp_wordweaver_supported_languages", $languages);
}

/**
 * Format file size
 */
function wp_wordweaver_format_bytes($size, $precision = 2)
{
    $units = ["B", "KB", "MB", "GB", "TB"];

    for ($i = 0; $size > 1024 && $i < count($units) - 1; $i++) {
        $size /= 1024;
    }

    return round($size, $precision) . " " . $units[$i];
}

/**
 * Check system requirements
 */
function wp_wordweaver_check_requirements()
{
    $requirements = [
        "php_version" => [
            "required" => "7.4",
            "current" => PHP_VERSION,
            "met" => version_compare(PHP_VERSION, "7.4", ">="),
        ],
        "wp_version" => [
            "required" => "5.0",
            "current" => get_bloginfo("version"),
            "met" => version_compare(get_bloginfo("version"), "5.0", ">="),
        ],
        "mbstring" => [
            "required" => true,
            "current" => extension_loaded("mbstring"),
            "met" => extension_loaded("mbstring"),
        ],
    ];

    return $requirements;
}

/**
 * Force reload text domain for immediate language switching
 */
function wp_wordweaver_reload_textdomain()
{
    global $l10n;

    // Remove existing textdomain
    if (isset($l10n["wp-wordweaver"])) {
        unset($l10n["wp-wordweaver"]);
    }

    // Reload textdomain
    $locale = get_locale();
    $mofile_local =
        WP_WORDWEAVER_PLUGIN_DIR . "languages/wp-wordweaver-{$locale}.mo";
    $mofile_global = WP_LANG_DIR . "/plugins/wp-wordweaver-{$locale}.mo";

    if (file_exists($mofile_global)) {
        load_textdomain("wp-wordweaver", $mofile_global);
    } elseif (file_exists($mofile_local)) {
        load_textdomain("wp-wordweaver", $mofile_local);
    }
}

/**
 * Check if current environment supports RTL
 */
function wp_wordweaver_is_rtl_locale()
{
    $locale = get_locale();
    $rtl_locales = ["fa", "fa_IR", "fa_AF", "ar", "he", "ur"];

    foreach ($rtl_locales as $rtl_locale) {
        if (strpos($locale, $rtl_locale) === 0) {
            return true;
        }
    }

    return false;
}

/**
 * Get admin body classes for WordWeaver
 */
function wp_wordweaver_get_admin_body_class()
{
    $classes = ["wp-wordweaver-admin"];

    if (wp_wordweaver_is_rtl_locale()) {
        $classes[] = "wp-wordweaver-rtl";
    }

    return implode(" ", $classes);
}

/**
 * Add RTL stylesheet if needed
 */
function wp_wordweaver_maybe_enqueue_rtl_styles()
{
    if (wp_wordweaver_is_rtl_locale()) {
        wp_enqueue_style(
            "wp-wordweaver-rtl",
            WP_WORDWEAVER_PLUGIN_URL . "assets/css/admin-rtl.css",
            ["wp-wordweaver-admin"],
            WP_WORDWEAVER_VERSION,
        );
    }
}
