<?php
/**
 * Plugin Name: WP WordWeaver
 * Plugin URI: https://wordpress.khaleghi.info/plugins/wp-wordweaver
 * Description: A powerful WordPress plugin for dynamic content manipulation and AI-powered content generation with bilingual support.
 * Version: 1.0.0
 * Author: Ali Khaleghi
 * Author URI: https://khaleghi.info
 * Text Domain: wp-wordweaver
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.4
 * Requires PHP: 7.4
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Network: false
 */

// Prevent direct access
if (!defined("ABSPATH")) {
    exit();
}

// Define plugin constants
define("WP_WORDWEAVER_VERSION", "1.0.0");
define("WP_WORDWEAVER_PLUGIN_DIR", plugin_dir_path(__FILE__));
define("WP_WORDWEAVER_PLUGIN_URL", plugin_dir_url(__FILE__));
define("WP_WORDWEAVER_PLUGIN_BASENAME", plugin_basename(__FILE__));

/**
 * Main WP WordWeaver Plugin Class
 */
class WP_WordWeaver
{
    /**
     * Single instance of the class
     */
    private static $instance = null;

    /**
     * Get single instance
     */
    public static function get_instance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct()
    {
        $this->init();
    }

    /**
     * Initialize the plugin
     */
    public function init()
    {
        // Load plugin textdomain for translations (early hook)
        add_action("init", [$this, "load_textdomain"], 1);

        // Include required files
        $this->include_files();

        // Initialize hooks
        $this->init_hooks();

        // Initialize classes
        $this->init_classes();
    }

    /**
     * Load plugin textdomain
     */
    public function load_textdomain()
    {
        $locale = get_locale();
        $mofile_local =
            WP_WORDWEAVER_PLUGIN_DIR . "languages/wp-wordweaver-{$locale}.mo";
        $mofile_global = WP_LANG_DIR . "/plugins/wp-wordweaver-{$locale}.mo";

        // Try to load from global first, then local
        if (file_exists($mofile_global)) {
            $loaded = load_textdomain("wp-wordweaver", $mofile_global);
        } elseif (file_exists($mofile_local)) {
            $loaded = load_textdomain("wp-wordweaver", $mofile_local);
        } else {
            $loaded = load_plugin_textdomain(
                "wp-wordweaver",
                false,
                dirname(WP_WORDWEAVER_PLUGIN_BASENAME) . "/languages/",
            );
        }

        // Add RTL support for Persian and other RTL languages
        if (wp_wordweaver_is_rtl_locale()) {
            add_action("admin_head", function () {
                echo '<style>
                    .wp-wordweaver-admin { direction: rtl; text-align: right; }
                    .wp-wordweaver-rtl * { font-family: Tahoma, "Iranian Sans", "B Nazanin", Arial, sans-serif; }
                </style>';
            });
        }

        // Hook to add plugin header translation
        add_filter("gettext", [$this, "translate_plugin_headers"], 10, 3);

        // Debug translation loading
        if (defined("WP_DEBUG") && WP_DEBUG) {
            error_log(
                "[WP WordWeaver] Textdomain loaded: " .
                    ($loaded ? "Yes" : "No"),
            );
            error_log("[WP WordWeaver] Current locale: " . $locale);
            error_log(
                "[WP WordWeaver] Checked files: " .
                    $mofile_global .
                    ", " .
                    $mofile_local,
            );
        }
    }

    /**
     * Include required files
     */
    private function include_files()
    {
        require_once WP_WORDWEAVER_PLUGIN_DIR . "includes/helpers.php";
        require_once WP_WORDWEAVER_PLUGIN_DIR . "includes/class-admin.php";
        require_once WP_WORDWEAVER_PLUGIN_DIR . "includes/class-processor.php";
        require_once WP_WORDWEAVER_PLUGIN_DIR . "includes/class-ai.php";
    }

    /**
     * Initialize hooks
     */
    private function init_hooks()
    {
        // Activation and deactivation hooks
        register_activation_hook(__FILE__, [$this, "activate"]);
        register_deactivation_hook(__FILE__, [$this, "deactivate"]);

        // Enqueue admin scripts and styles
        add_action("admin_enqueue_scripts", [$this, "admin_enqueue_scripts"]);
    }

    /**
     * Initialize classes
     */
    private function init_classes()
    {
        if (is_admin()) {
            WP_WordWeaver_Admin::get_instance();
        }
        WP_WordWeaver_Processor::get_instance();
        WP_WordWeaver_AI::get_instance();
    }

    /**
     * Plugin activation
     */
    public function activate()
    {
        // Create necessary database tables or options
        $this->create_options();

        // Flush rewrite rules if needed
        flush_rewrite_rules();
    }

    /**
     * Plugin deactivation
     */
    public function deactivate()
    {
        // Cleanup if necessary
        flush_rewrite_rules();
    }

    /**
     * Create default options
     */
    private function create_options()
    {
        $default_options = [
            "enabled" => true,
            "language" => "en",
            "auto_process" => false,
            "ai_enabled" => false,
        ];

        add_option("wp_wordweaver_options", $default_options);
    }

    /**
     * Enqueue admin scripts and styles
     */
    public function admin_enqueue_scripts($hook)
    {
        // Only load on our admin pages
        if (strpos($hook, "wp-wordweaver") !== false) {
            wp_enqueue_style(
                "wp-wordweaver-admin",
                WP_WORDWEAVER_PLUGIN_URL . "assets/css/admin.css",
                [],
                WP_WORDWEAVER_VERSION,
            );

            // Load RTL styles if needed
            wp_wordweaver_maybe_enqueue_rtl_styles();

            wp_enqueue_script(
                "wp-wordweaver-admin",
                WP_WORDWEAVER_PLUGIN_URL . "assets/js/admin.js",
                ["jquery"],
                WP_WORDWEAVER_VERSION,
                true,
            );

            // Add admin body classes for RTL support
            add_filter("admin_body_class", function ($classes) {
                return $classes . " " . wp_wordweaver_get_admin_body_class();
            });

            // Localize script for AJAX
            wp_localize_script("wp-wordweaver-admin", "wpWordWeaver", [
                "ajaxUrl" => admin_url("admin-ajax.php"),
                "nonce" => wp_create_nonce("wp_wordweaver_nonce"),
                "isRTL" => wp_wordweaver_is_rtl_locale(),
                "locale" => get_locale(),
                "strings" => [
                    "processing" => __("Processing...", "wp-wordweaver"),
                    "success" => __("Success!", "wp-wordweaver"),
                    "error" => __("Error occurred", "wp-wordweaver"),
                ],
            ]);
        }
    }

    /**
     * Translate plugin headers
     */
    public function translate_plugin_headers($translated, $text, $domain)
    {
        if ($domain !== "wp-wordweaver") {
            return $translated;
        }

        // Translate plugin name in plugin list
        if ($text === "WP WordWeaver") {
            return wp_wordweaver_get_translated_plugin_name();
        }

        return $translated;
    }
}

// Initialize the plugin
WP_WordWeaver::get_instance();

// Add plugin action links for translation
add_filter("plugin_action_links_" . WP_WORDWEAVER_PLUGIN_BASENAME, function (
    $links,
) {
    $settings_link =
        '<a href="' .
        admin_url("admin.php?page=wp-wordweaver-settings") .
        '">' .
        __("Settings", "wp-wordweaver") .
        "</a>";
    array_unshift($links, $settings_link);
    return $links;
});

// Add plugin row meta for translation
add_filter(
    "plugin_row_meta",
    function ($links, $file) {
        if ($file === WP_WORDWEAVER_PLUGIN_BASENAME) {
            $new_links = [
                '<a href="' .
                admin_url("admin.php?page=wp-wordweaver") .
                '">' .
                __("Dashboard", "wp-wordweaver") .
                "</a>",
                '<a href="' .
                admin_url("admin.php?page=wp-wordweaver-processor") .
                '">' .
                __("Process Content", "wp-wordweaver") .
                "</a>",
            ];
            $links = array_merge($links, $new_links);
        }
        return $links;
    },
    10,
    2,
);
